<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn() || !isBranchManager()) {
    setFlashMessage('error', 'Access denied. Branch manager privileges required.');
    redirect('/auth/login.php');
}

$db = getDB();
$userId = getCurrentUserId();
$branchId = getCurrentUserBranchId();

// Get branch info
$branchStmt = $db->prepare("SELECT * FROM branches WHERE id = ?");
$branchStmt->execute([$branchId]);
$branch = $branchStmt->fetch();

// Get stats for this branch
$stats = [];

// Total complaints for this branch
$stmt = $db->prepare("SELECT COUNT(*) as count FROM complaints WHERE branch_id = ?");
$stmt->execute([$branchId]);
$stats['total'] = $stmt->fetch()['count'];

// Open complaints
$stmt = $db->prepare("SELECT COUNT(*) as count FROM complaints WHERE branch_id = ? AND status = 'open'");
$stmt->execute([$branchId]);
$stats['open'] = $stmt->fetch()['count'];

// In progress
$stmt = $db->prepare("SELECT COUNT(*) as count FROM complaints WHERE branch_id = ? AND status = 'in_progress'");
$stmt->execute([$branchId]);
$stats['in_progress'] = $stmt->fetch()['count'];

// Resolved
$stmt = $db->prepare("SELECT COUNT(*) as count FROM complaints WHERE branch_id = ? AND status IN ('resolved', 'closed')");
$stmt->execute([$branchId]);
$stats['resolved'] = $stmt->fetch()['count'];

// Urgent complaints
$stmt = $db->prepare("SELECT COUNT(*) as count FROM complaints WHERE branch_id = ? AND priority = 'urgent' AND status NOT IN ('resolved', 'closed')");
$stmt->execute([$branchId]);
$stats['urgent'] = $stmt->fetch()['count'];

// Recent complaints for this branch
$recentComplaints = $db->prepare("
    SELECT c.*, o.order_number, u.whatsapp_number as customer_whatsapp,
           i.title as issue_title
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN users u ON c.customer_id = u.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    WHERE c.branch_id = ?
    ORDER BY c.created_at DESC
    LIMIT 10
");
$recentComplaints->execute([$branchId]);
$recentComplaints = $recentComplaints->fetchAll();

$pageTitle = 'Branch Manager Dashboard - ' . ($branch['name'] ?? 'Support Center');
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-6">
    <div class="flex items-center justify-between flex-wrap gap-4">
        <div>
            <h1 class="text-2xl font-bold text-gray-800">Branch Dashboard</h1>
            <p class="text-gray-500 flex items-center gap-2 mt-1">
                <i class="fas fa-store"></i>
                <?= sanitize($branch['name'] ?? 'Unknown Branch') ?>
            </p>
        </div>
        <a href="<?= BASE_URL ?>/manager/complaints.php" class="gradient-bg text-white px-4 py-2 rounded-lg font-medium hover:opacity-90 transition">
            <i class="fas fa-list mr-2"></i>View All Complaints
        </a>
    </div>
</div>

<!-- Stats Cards -->
<div class="grid grid-cols-2 md:grid-cols-5 gap-4 mb-8">
    <div class="bg-white rounded-xl shadow-lg p-4 md:p-6">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-gray-500 text-sm">Total</p>
                <p class="text-2xl md:text-3xl font-bold text-gray-800"><?= $stats['total'] ?></p>
            </div>
            <div class="w-10 h-10 md:w-12 md:h-12 bg-blue-100 rounded-full flex items-center justify-center">
                <i class="fas fa-ticket-alt text-blue-500 text-lg md:text-xl"></i>
            </div>
        </div>
    </div>
    <div class="bg-white rounded-xl shadow-lg p-4 md:p-6">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-gray-500 text-sm">Open</p>
                <p class="text-2xl md:text-3xl font-bold text-blue-600"><?= $stats['open'] ?></p>
            </div>
            <div class="w-10 h-10 md:w-12 md:h-12 bg-blue-100 rounded-full flex items-center justify-center">
                <i class="fas fa-folder-open text-blue-500 text-lg md:text-xl"></i>
            </div>
        </div>
    </div>
    <div class="bg-white rounded-xl shadow-lg p-4 md:p-6">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-gray-500 text-sm">In Progress</p>
                <p class="text-2xl md:text-3xl font-bold text-yellow-600"><?= $stats['in_progress'] ?></p>
            </div>
            <div class="w-10 h-10 md:w-12 md:h-12 bg-yellow-100 rounded-full flex items-center justify-center">
                <i class="fas fa-spinner text-yellow-500 text-lg md:text-xl"></i>
            </div>
        </div>
    </div>
    <div class="bg-white rounded-xl shadow-lg p-4 md:p-6">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-gray-500 text-sm">Resolved</p>
                <p class="text-2xl md:text-3xl font-bold text-green-600"><?= $stats['resolved'] ?></p>
            </div>
            <div class="w-10 h-10 md:w-12 md:h-12 bg-green-100 rounded-full flex items-center justify-center">
                <i class="fas fa-check-circle text-green-500 text-lg md:text-xl"></i>
            </div>
        </div>
    </div>
    <div class="bg-white rounded-xl shadow-lg p-4 md:p-6 col-span-2 md:col-span-1">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-gray-500 text-sm">Urgent</p>
                <p class="text-2xl md:text-3xl font-bold text-red-600"><?= $stats['urgent'] ?></p>
            </div>
            <div class="w-10 h-10 md:w-12 md:h-12 bg-red-100 rounded-full flex items-center justify-center">
                <i class="fas fa-exclamation-triangle text-red-500 text-lg md:text-xl"></i>
            </div>
        </div>
    </div>
</div>

<!-- Recent Complaints -->
<div class="bg-white rounded-xl shadow-lg overflow-hidden">
    <div class="p-4 md:p-6 border-b">
        <h2 class="text-lg font-semibold text-gray-800">Recent Complaints</h2>
    </div>
    
    <?php if (empty($recentComplaints)): ?>
    <div class="text-center py-12 text-gray-500">
        <i class="fas fa-inbox text-5xl mb-4"></i>
        <p class="text-lg">No complaints for this branch yet</p>
    </div>
    <?php else: ?>
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ticket</th>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">Issue</th>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Customer</th>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">Priority</th>
                    <th class="px-4 md:px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Action</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($recentComplaints as $complaint): ?>
                <tr class="hover:bg-gray-50 transition">
                    <td class="px-4 md:px-6 py-4">
                        <span class="font-mono text-sm"><?= sanitize($complaint['ticket_number']) ?></span>
                        <p class="text-xs text-gray-500 md:hidden mt-1"><?= sanitize($complaint['issue_title'] ?? 'N/A') ?></p>
                    </td>
                    <td class="px-4 md:px-6 py-4 hidden md:table-cell">
                        <span class="text-gray-800"><?= sanitize($complaint['issue_title'] ?? 'N/A') ?></span>
                    </td>
                    <td class="px-4 md:px-6 py-4">
                        <a href="https://wa.me/<?= preg_replace('/[^0-9]/', '', $complaint['customer_whatsapp']) ?>" 
                           target="_blank" class="text-green-600 hover:underline flex items-center gap-1">
                            <i class="fab fa-whatsapp"></i>
                            <span class="hidden sm:inline"><?= sanitize($complaint['customer_whatsapp']) ?></span>
                            <span class="sm:hidden">Chat</span>
                        </a>
                    </td>
                    <td class="px-4 md:px-6 py-4">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getStatusBadgeClass($complaint['status']) ?>">
                            <?= ucfirst(str_replace('_', ' ', $complaint['status'])) ?>
                        </span>
                    </td>
                    <td class="px-4 md:px-6 py-4 hidden lg:table-cell">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getPriorityBadgeClass($complaint['priority']) ?>">
                            <?= ucfirst($complaint['priority']) ?>
                        </span>
                    </td>
                    <td class="px-4 md:px-6 py-4 text-right">
                        <a href="<?= BASE_URL ?>/complaint/view.php?ticket=<?= sanitize($complaint['ticket_number']) ?>" 
                           class="text-primary hover:underline">
                            <i class="fas fa-eye"></i>
                            <span class="hidden sm:inline ml-1">View</span>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
