-- Customer Feedback and Complaint Management System Database Schema

CREATE DATABASE IF NOT EXISTS complaint_management;
USE complaint_management;

-- Branches table (physical locations/stores)
CREATE TABLE branches (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    code VARCHAR(20) NOT NULL UNIQUE,
    address TEXT,
    phone VARCHAR(20),
    email VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Departments table (internal departments for routing)
CREATE TABLE departments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    email VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Users table with roles (admin, branch_manager, staff, customer)
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NULL,
    password VARCHAR(255) NULL,
    full_name VARCHAR(100) NULL,
    whatsapp_number VARCHAR(20) NOT NULL UNIQUE,
    role ENUM('admin', 'branch_manager', 'staff', 'customer') DEFAULT 'customer',
    branch_id INT NULL,
    department_id INT NULL,
    is_verified BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE SET NULL,
    FOREIGN KEY (department_id) REFERENCES departments(id) ON DELETE SET NULL
);

-- OTP verification table
CREATE TABLE otp_verifications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    whatsapp_number VARCHAR(20) NOT NULL,
    otp_code VARCHAR(6) NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    is_used BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_whatsapp (whatsapp_number),
    INDEX idx_expires (expires_at)
);

-- Orders table
CREATE TABLE orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_number VARCHAR(50) NOT NULL UNIQUE,
    branch_id INT NOT NULL,
    customer_id INT NULL,
    customer_name VARCHAR(100) NULL,
    order_date DATE NOT NULL,
    total_amount DECIMAL(10, 2) NOT NULL,
    status ENUM('pending', 'processing', 'shipped', 'delivered', 'cancelled') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE CASCADE,
    FOREIGN KEY (customer_id) REFERENCES users(id) ON DELETE SET NULL
);

-- Complaint issues (detailed single selection replacing category + subject)
CREATE TABLE complaint_issues (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    department_id INT,
    priority ENUM('low', 'medium', 'high', 'urgent') DEFAULT 'medium',
    is_active BOOLEAN DEFAULT TRUE,
    display_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (department_id) REFERENCES departments(id) ON DELETE SET NULL
);

-- Complaints table
CREATE TABLE complaints (
    id INT AUTO_INCREMENT PRIMARY KEY,
    ticket_number VARCHAR(20) NOT NULL UNIQUE,
    access_token VARCHAR(64) NOT NULL UNIQUE,
    order_id INT NOT NULL UNIQUE,
    branch_id INT NOT NULL,
    customer_id INT NOT NULL,
    issue_id INT NOT NULL,
    department_id INT,
    additional_details TEXT NULL,
    priority ENUM('low', 'medium', 'high', 'urgent') DEFAULT 'medium',
    status ENUM('open', 'in_progress', 'pending_customer', 'resolved', 'closed') DEFAULT 'open',
    assigned_to INT NULL,
    closed_by INT NULL,
    closed_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
    FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE CASCADE,
    FOREIGN KEY (customer_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (issue_id) REFERENCES complaint_issues(id) ON DELETE CASCADE,
    FOREIGN KEY (department_id) REFERENCES departments(id) ON DELETE SET NULL,
    FOREIGN KEY (assigned_to) REFERENCES users(id) ON DELETE SET NULL,
    FOREIGN KEY (closed_by) REFERENCES users(id) ON DELETE SET NULL
);

-- WhatsApp message log (for tracking all messages sent)
CREATE TABLE whatsapp_messages (
    id INT AUTO_INCREMENT PRIMARY KEY,
    complaint_id INT NULL,
    recipient_number VARCHAR(20) NOT NULL,
    message_type ENUM('otp', 'complaint_receipt', 'manager_notification', 'customer_update', 'other') NOT NULL,
    message_content TEXT NOT NULL,
    sent_by INT NULL,
    is_sent BOOLEAN DEFAULT FALSE,
    sent_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (complaint_id) REFERENCES complaints(id) ON DELETE SET NULL,
    FOREIGN KEY (sent_by) REFERENCES users(id) ON DELETE SET NULL
);

-- Complaint responses/comments
CREATE TABLE complaint_responses (
    id INT AUTO_INCREMENT PRIMARY KEY,
    complaint_id INT NOT NULL,
    user_id INT NOT NULL,
    message TEXT NOT NULL,
    is_internal BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (complaint_id) REFERENCES complaints(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Complaint attachments
CREATE TABLE complaint_attachments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    complaint_id INT NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_type VARCHAR(100),
    file_size INT,
    uploaded_by INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (complaint_id) REFERENCES complaints(id) ON DELETE CASCADE,
    FOREIGN KEY (uploaded_by) REFERENCES users(id) ON DELETE CASCADE
);

-- Activity log for tracking actions
CREATE TABLE activity_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    complaint_id INT,
    user_id INT,
    action VARCHAR(100) NOT NULL,
    details TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (complaint_id) REFERENCES complaints(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

-- Insert sample departments
INSERT INTO departments (name, description, email) VALUES
('Customer Service', 'General customer inquiries and support', 'support@company.com'),
('Technical Support', 'Technical issues and product support', 'tech@company.com'),
('Billing', 'Payment and billing related issues', 'billing@company.com'),
('Shipping', 'Delivery and shipping concerns', 'shipping@company.com'),
('Quality Assurance', 'Product quality complaints', 'quality@company.com');

-- Insert sample branches
INSERT INTO branches (name, code, address, phone, email) VALUES
('Main Branch - Downtown', 'MAIN-001', '123 Main Street, Downtown, City 10001', '+1-555-100-0001', 'main@company.com'),
('North Branch', 'NORTH-002', '456 North Avenue, Northside, City 10002', '+1-555-100-0002', 'north@company.com'),
('South Branch', 'SOUTH-003', '789 South Boulevard, Southside, City 10003', '+1-555-100-0003', 'south@company.com'),
('East Mall Outlet', 'EAST-004', 'East Shopping Mall, Unit 45, City 10004', '+1-555-100-0004', 'east@company.com'),
('West Plaza Store', 'WEST-005', 'West Plaza, Shop 12, City 10005', '+1-555-100-0005', 'west@company.com');

-- Insert detailed complaint issues (single selection replacing category + subject)
INSERT INTO complaint_issues (title, description, department_id, priority, display_order) VALUES
('Product arrived damaged or broken', 'The product I received is physically damaged, broken, or has visible defects', 5, 'high', 1),
('Received wrong product/item', 'The product delivered is different from what I ordered', 4, 'high', 2),
('Product missing from order', 'One or more items from my order are missing', 4, 'high', 3),
('Order not delivered yet', 'My order has not been delivered within the expected timeframe', 4, 'medium', 4),
('Product quality not as expected', 'The product quality does not match the description or my expectations', 5, 'medium', 5),
('Incorrect billing/overcharged', 'I was charged incorrectly or more than the actual amount', 3, 'high', 6),
('Refund not received', 'I returned the product but have not received my refund yet', 3, 'high', 7),
('Product not working/defective', 'The product is not functioning properly or has technical issues', 2, 'high', 8),
('Poor customer service experience', 'I had an unsatisfactory experience with staff or service', 1, 'medium', 9),
('Packaging was poor/inadequate', 'The product packaging was insufficient causing potential damage', 4, 'low', 10),
('Invoice/receipt issue', 'Problem with invoice, receipt, or billing documentation', 3, 'low', 11),
('Request for exchange', 'I would like to exchange my product for a different one', 1, 'medium', 12),
('Warranty claim', 'I need to claim warranty for my product', 2, 'medium', 13),
('Delivery person behavior', 'Issue with the delivery personnel conduct or behavior', 4, 'medium', 14),
('Other issue not listed above', 'My issue is not covered by the options above', 1, 'medium', 15);

-- Insert admin user (password: password) - Admin doesn't need branch
INSERT INTO users (email, password, full_name, whatsapp_number, role, is_verified) VALUES
('admin@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'System Administrator', '+1234567890', 'admin', TRUE);

-- Insert branch managers (password: password) - Each branch has a manager
INSERT INTO users (email, password, full_name, whatsapp_number, role, branch_id, is_verified) VALUES
('manager.main@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Main Branch Manager', '+1234567801', 'branch_manager', 1, TRUE),
('manager.north@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'North Branch Manager', '+1234567802', 'branch_manager', 2, TRUE),
('manager.south@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'South Branch Manager', '+1234567803', 'branch_manager', 3, TRUE),
('manager.east@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'East Mall Manager', '+1234567804', 'branch_manager', 4, TRUE),
('manager.west@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'West Plaza Manager', '+1234567805', 'branch_manager', 5, TRUE);

-- Insert sample staff users (password: password) - Staff must have branch
INSERT INTO users (email, password, full_name, whatsapp_number, role, branch_id, department_id, is_verified) VALUES
('support@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Support Agent', '+1234567891', 'staff', 1, 1, TRUE),
('tech@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Tech Support', '+1234567892', 'staff', 1, 2, TRUE);

-- Insert sample customers (verified via WhatsApp) - Customers don't have branch
INSERT INTO users (email, full_name, whatsapp_number, role, is_verified) VALUES
('john@example.com', 'John Doe', '+15551234567', 'customer', TRUE),
('jane@example.com', 'Jane Smith', '+15559876543', 'customer', TRUE);

-- Insert sample orders (with branch_id)
INSERT INTO orders (order_number, branch_id, customer_id, customer_name, order_date, total_amount, status) VALUES
('ORD-2024-001', 1, 4, 'John Doe', '2024-01-15', 299.99, 'delivered'),
('ORD-2024-002', 2, 4, 'John Doe', '2024-01-20', 149.50, 'shipped'),
('ORD-2024-003', 1, 5, 'Jane Smith', '2024-01-22', 599.00, 'processing'),
('ORD-2024-004', 3, 5, 'Jane Smith', '2024-01-25', 89.99, 'delivered'),
('ORD-2024-005', 4, NULL, 'Guest Customer', '2024-01-26', 199.00, 'delivered'),
('ORD-2024-006', 5, NULL, 'Walk-in Customer', '2024-01-27', 75.50, 'delivered');
