<?php
require_once __DIR__ . '/../config/config.php';

if (!isCustomerLoggedIn()) {
    redirect('/');
}

$db = getDB();
$userId = getCurrentUserId();

// Filters
$statusFilter = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';

$where = "WHERE c.customer_id = ?";
$params = [$userId];

if ($statusFilter) {
    $where .= " AND c.status = ?";
    $params[] = $statusFilter;
}

if ($search) {
    $where .= " AND (c.ticket_number LIKE ? OR i.title LIKE ? OR o.order_number LIKE ?)";
    $searchTerm = "%$search%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 10;
$offset = ($page - 1) * $perPage;

// Get total count
$countStmt = $db->prepare("SELECT COUNT(*) as total FROM complaints c JOIN orders o ON c.order_id = o.id LEFT JOIN complaint_issues i ON c.issue_id = i.id $where");
$countStmt->execute($params);
$totalCount = $countStmt->fetch()['total'];
$totalPages = ceil($totalCount / $perPage);

// Get complaints
$stmt = $db->prepare("
    SELECT c.*, o.order_number, i.title as issue_title, d.name as department_name, b.name as branch_name
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN branches b ON c.branch_id = b.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    LEFT JOIN departments d ON c.department_id = d.id
    $where
    ORDER BY c.created_at DESC
    LIMIT $perPage OFFSET $offset
");
$stmt->execute($params);
$complaints = $stmt->fetchAll();

$pageTitle = 'My Complaints - Support Center';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-6">
    <h1 class="text-2xl font-bold text-gray-800">My Complaints</h1>
    <p class="text-gray-500">View and manage all your submitted complaints</p>
</div>

<!-- Filters -->
<div class="bg-white rounded-xl shadow-lg p-4 mb-6">
    <form method="GET" class="flex flex-wrap items-center gap-4">
        <div class="flex-1 min-w-[200px]">
            <div class="relative">
                <span class="absolute inset-y-0 left-0 pl-3 flex items-center text-gray-400">
                    <i class="fas fa-search"></i>
                </span>
                <input type="text" name="search" value="<?= sanitize($search) ?>"
                    class="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                    placeholder="Search by ticket, subject, or order...">
            </div>
        </div>
        <select name="status" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
            <option value="">All Status</option>
            <option value="open" <?= $statusFilter === 'open' ? 'selected' : '' ?>>Open</option>
            <option value="in_progress" <?= $statusFilter === 'in_progress' ? 'selected' : '' ?>>In Progress</option>
            <option value="pending_customer" <?= $statusFilter === 'pending_customer' ? 'selected' : '' ?>>Pending Customer</option>
            <option value="resolved" <?= $statusFilter === 'resolved' ? 'selected' : '' ?>>Resolved</option>
            <option value="closed" <?= $statusFilter === 'closed' ? 'selected' : '' ?>>Closed</option>
        </select>
        <button type="submit" class="gradient-bg text-white px-6 py-2 rounded-lg font-medium hover:opacity-90 transition">
            <i class="fas fa-filter mr-2"></i>Filter
        </button>
        <?php if ($search || $statusFilter): ?>
        <a href="<?= BASE_URL ?>/customer/complaints.php" class="text-gray-500 hover:text-gray-700">
            <i class="fas fa-times mr-1"></i>Clear
        </a>
        <?php endif; ?>
    </form>
</div>

<!-- Complaints List -->
<div class="bg-white rounded-xl shadow-lg overflow-hidden">
    <?php if (empty($complaints)): ?>
    <div class="text-center py-12 text-gray-500">
        <i class="fas fa-inbox text-5xl mb-4"></i>
        <p class="text-lg">No complaints found</p>
        <p class="text-sm mt-2">You haven't submitted any complaints yet.</p>
    </div>
    <?php else: ?>
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ticket</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Issue</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden md:table-cell">Branch</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden sm:table-cell">Priority</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden lg:table-cell">Date</th>
                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Action</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($complaints as $complaint): ?>
                <tr class="hover:bg-gray-50 transition">
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="font-mono text-sm text-primary"><?= sanitize($complaint['ticket_number']) ?></span>
                    </td>
                    <td class="px-6 py-4">
                        <p class="text-gray-800 font-medium line-clamp-1 max-w-xs"><?= sanitize($complaint['issue_title'] ?? 'Complaint') ?></p>
                        <p class="text-xs text-gray-500"><?= sanitize($complaint['order_number']) ?></p>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap hidden md:table-cell">
                        <span class="text-sm text-gray-600"><?= sanitize($complaint['branch_name']) ?></span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getStatusBadgeClass($complaint['status']) ?>">
                            <?= ucfirst(str_replace('_', ' ', $complaint['status'])) ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap hidden sm:table-cell">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getPriorityBadgeClass($complaint['priority']) ?>">
                            <?= ucfirst($complaint['priority']) ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 hidden lg:table-cell">
                        <?= date('M d, Y', strtotime($complaint['created_at'])) ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-right">
                        <a href="<?= BASE_URL ?>/complaint/view.php?ticket=<?= sanitize($complaint['ticket_number']) ?>" 
                           class="text-primary hover:text-secondary font-medium text-sm">
                            View <i class="fas fa-arrow-right ml-1"></i>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
    <div class="px-6 py-4 border-t flex items-center justify-between">
        <p class="text-sm text-gray-500">
            Showing <?= $offset + 1 ?> to <?= min($offset + $perPage, $totalCount) ?> of <?= $totalCount ?> results
        </p>
        <div class="flex gap-2">
            <?php if ($page > 1): ?>
            <a href="?page=<?= $page - 1 ?>&status=<?= urlencode($statusFilter) ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded hover:bg-gray-50 transition">
                <i class="fas fa-chevron-left"></i>
            </a>
            <?php endif; ?>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
            <a href="?page=<?= $i ?>&status=<?= urlencode($statusFilter) ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded <?= $i === $page ? 'bg-primary text-white' : 'hover:bg-gray-50' ?> transition">
                <?= $i ?>
            </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
            <a href="?page=<?= $page + 1 ?>&status=<?= urlencode($statusFilter) ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded hover:bg-gray-50 transition">
                <i class="fas fa-chevron-right"></i>
            </a>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
