<?php
require_once __DIR__ . '/../config/config.php';

$ticketNumber = $_GET['ticket'] ?? '';
$accessToken = $_GET['token'] ?? '';

// Must have either ticket number or access token
if (empty($ticketNumber) && empty($accessToken)) {
    setFlashMessage('error', 'Invalid access. Please use the link from your WhatsApp message.');
    redirect('/');
}

$db = getDB();

// Fetch complaint with related data - by ticket number or access token
$query = "
    SELECT c.*, 
           o.order_number, o.order_date, o.total_amount, o.status as order_status,
           u.whatsapp_number as customer_whatsapp, u.email as customer_email,
           i.title as issue_title, i.description as issue_description,
           b.name as branch_name, b.address as branch_address,
           d.name as department_name,
           assigned.full_name as assigned_to_name,
           closer.full_name as closed_by_name
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN users u ON c.customer_id = u.id
    JOIN branches b ON c.branch_id = b.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    LEFT JOIN departments d ON c.department_id = d.id
    LEFT JOIN users assigned ON c.assigned_to = assigned.id
    LEFT JOIN users closer ON c.closed_by = closer.id
";

if (!empty($accessToken)) {
    $query .= " WHERE c.access_token = ?";
    $stmt = $db->prepare($query);
    $stmt->execute([$accessToken]);
} else {
    $query .= " WHERE c.ticket_number = ?";
    $stmt = $db->prepare($query);
    $stmt->execute([$ticketNumber]);
}
$complaint = $stmt->fetch();

if (!$complaint) {
    setFlashMessage('error', 'Complaint not found.');
    redirect('/');
}

// Access via token grants view and respond access (for customers without login)
$accessViaToken = !empty($accessToken) && $complaint['access_token'] === $accessToken;

// Check access: token holder, customer owner, admin, branch manager, or assigned staff can view
$canView = false;
$canClose = false;
$canRespond = false;
$userId = null;
$role = null;

if ($accessViaToken) {
    // Token access - customer can view, respond, and close
    $canView = true;
    $canClose = ($complaint['status'] !== 'closed'); // Can't reopen closed complaints
    $canRespond = ($complaint['status'] !== 'closed');
    $userId = $complaint['customer_id']; // Use customer ID for responses
}

if (isLoggedIn()) {
    $userId = getCurrentUserId();
    $role = getCurrentUserRole();

    if ($role === 'admin') {
        $canView = true;
        $canClose = true;
        $canRespond = true;
    } elseif ($role === 'branch_manager' && $complaint['branch_id'] == getCurrentUserBranchId()) {
        $canView = true;
        $canClose = true;
        $canRespond = true;
    } elseif ($role === 'staff' && $complaint['assigned_to'] == $userId) {
        $canView = true;
        $canRespond = true;
    } elseif ($complaint['customer_id'] == $userId) {
        $canView = true;
        $canClose = ($complaint['status'] !== 'closed'); // Can't reopen closed complaints
        $canRespond = ($complaint['status'] !== 'closed');
    }
}

if (!$canView) {
    setFlashMessage('error', 'You do not have permission to view this complaint.');
    redirect('/');
}

// Prevent any actions on closed complaints (except viewing)
$isClosed = ($complaint['status'] === 'closed');
if ($isClosed) {
    $canClose = false;
    $canRespond = false;
}

// Handle response submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $canRespond) {
    $action = $_POST['action'] ?? '';

    if ($action === 'respond') {
        $message = sanitize($_POST['message'] ?? '');
        $isInternal = isset($_POST['is_internal']) && isStaff();

        if (!empty($message)) {
            $stmt = $db->prepare("INSERT INTO complaint_responses (complaint_id, user_id, message, is_internal) VALUES (?, ?, ?, ?)");
            $stmt->execute([$complaint['id'], $userId, $message, $isInternal ? 1 : 0]);

            // Log activity
            $logStmt = $db->prepare("INSERT INTO activity_logs (complaint_id, user_id, action, details) VALUES (?, ?, ?, ?)");
            $logStmt->execute([$complaint['id'], $userId, 'responded', 'Added a response']);

            // Send WhatsApp notification to customer if not internal note
            if (!$isInternal && !empty($complaint['customer_whatsapp'])) {
                $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
                $trackingLink = $protocol . $_SERVER['HTTP_HOST'] . BASE_URL . '/complaint/view.php?token=' . $complaint['access_token'];
                $responderName = $_SESSION['user_name'] ?? 'Support Team';
                
                $whatsappMessage = "🔔 *Update on Your Complaint*\n\n";
                $whatsappMessage .= "Ticket: *{$complaint['ticket_number']}*\n";
                $whatsappMessage .= "Order: {$complaint['order_number']}\n\n";
                $whatsappMessage .= "📝 *Response from {$responderName}:*\n";
                $whatsappMessage .= $message . "\n\n";
                $whatsappMessage .= "📱 View & Reply:\n";
                $whatsappMessage .= "{$trackingLink}\n\n";
                $whatsappMessage .= "_Thank you for your patience!_";
                
                $whatsappResult = WhatsAppOTP::sendMessage($complaint['customer_whatsapp'], $whatsappMessage);
                
                // Log WhatsApp message
                logWhatsAppMessage(
                    $db,
                    $complaint['id'],
                    $complaint['customer_whatsapp'],
                    'customer_update',
                    $whatsappMessage,
                    $userId,
                    $whatsappResult['success']
                );
            }

            setFlashMessage('success', 'Response added successfully.' . (!$isInternal ? ' Customer notified via WhatsApp.' : ''));
            redirect('/complaint/view.php?ticket=' . $ticketNumber);
        }
    } elseif ($action === 'close' && $canClose) {
        // Only customer owner or admin can close
        $stmt = $db->prepare("UPDATE complaints SET status = 'closed', closed_by = ?, closed_at = NOW() WHERE id = ?");
        $stmt->execute([$userId, $complaint['id']]);

        // Log activity
        $logStmt = $db->prepare("INSERT INTO activity_logs (complaint_id, user_id, action, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$complaint['id'], $userId, 'closed', 'Complaint closed']);

        setFlashMessage('success', 'Complaint has been closed.');
        redirect('/complaint/view.php?ticket=' . $ticketNumber);
    } elseif ($action === 'reopen' && ($complaint['customer_id'] == $userId || isAdmin())) {
        $stmt = $db->prepare("UPDATE complaints SET status = 'open', closed_by = NULL, closed_at = NULL WHERE id = ?");
        $stmt->execute([$complaint['id']]);

        // Log activity
        $logStmt = $db->prepare("INSERT INTO activity_logs (complaint_id, user_id, action, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$complaint['id'], $userId, 'reopened', 'Complaint reopened']);

        setFlashMessage('success', 'Complaint has been reopened.');
        redirect('/complaint/view.php?ticket=' . $ticketNumber);
    }
}

// Fetch responses
$responses = $db->prepare("
    SELECT r.*, u.full_name, u.role
    FROM complaint_responses r
    JOIN users u ON r.user_id = u.id
    WHERE r.complaint_id = ?
    ORDER BY r.created_at ASC
");
$responses->execute([$complaint['id']]);
$responses = $responses->fetchAll();

// Fetch attachments
$attachments = $db->prepare("SELECT * FROM complaint_attachments WHERE complaint_id = ?");
$attachments->execute([$complaint['id']]);
$attachments = $attachments->fetchAll();

// Fetch activity log
$activities = $db->prepare("
    SELECT a.*, u.full_name
    FROM activity_logs a
    LEFT JOIN users u ON a.user_id = u.id
    WHERE a.complaint_id = ?
    ORDER BY a.created_at DESC
    LIMIT 10
");
$activities->execute([$complaint['id']]);
$activities = $activities->fetchAll();

$pageTitle = 'Ticket ' . $ticketNumber . ' - Support Center';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="max-w-5xl mx-auto">
    <!-- Breadcrumb -->
    <nav class="mb-6">
        <ol class="flex items-center space-x-2 text-sm text-gray-500">
            <li><a href="<?= BASE_URL ?>/" class="hover:text-primary">Home</a></li>
            <li><i class="fas fa-chevron-right text-xs"></i></li>
            <li><a href="<?= BASE_URL ?>/customer/complaints.php" class="hover:text-primary">My Complaints</a></li>
            <li><i class="fas fa-chevron-right text-xs"></i></li>
            <li class="text-gray-800 font-medium"><?= sanitize($ticketNumber) ?></li>
        </ol>
    </nav>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <!-- Main Content -->
        <div class="lg:col-span-2 space-y-6">
            <!-- Complaint Header -->
            <div class="bg-white rounded-xl shadow-lg p-6">
                <div class="flex items-start justify-between flex-wrap gap-4 mb-4">
                    <div>
                        <div class="flex items-center gap-3 mb-2">
                            <span class="px-3 py-1 rounded-full text-sm font-medium <?= getStatusBadgeClass($complaint['status']) ?>">
                                <?= ucfirst(str_replace('_', ' ', $complaint['status'])) ?>
                            </span>
                            <span class="px-3 py-1 rounded-full text-sm font-medium <?= getPriorityBadgeClass($complaint['priority']) ?>">
                                <?= ucfirst($complaint['priority']) ?> Priority
                            </span>
                        </div>
                        <h1 class="text-2xl font-bold text-gray-800"><?= sanitize($complaint['issue_title'] ?? 'Complaint') ?></h1>
                    </div>
                    <div class="text-right">
                        <p class="text-sm text-gray-500">Ticket #</p>
                        <p class="font-mono font-semibold text-primary"><?= sanitize($ticketNumber) ?></p>
                    </div>
                </div>

                <?php if ($complaint['issue_description']): ?>
                <div class="prose max-w-none text-gray-600 bg-gray-50 rounded-lg p-4 mb-4">
                    <p class="text-sm"><i class="fas fa-info-circle mr-1"></i><?= sanitize($complaint['issue_description']) ?></p>
                </div>
                <?php endif; ?>
                
                <?php if ($complaint['additional_details']): ?>
                <div class="prose max-w-none text-gray-700 bg-blue-50 rounded-lg p-4">
                    <p class="text-sm font-medium text-gray-600 mb-2">Additional Details from Customer:</p>
                    <?= nl2br(sanitize($complaint['additional_details'])) ?>
                </div>
                <?php endif; ?>

                <?php if (!empty($attachments)): ?>
                <div class="mt-4 pt-4 border-t">
                    <h3 class="text-sm font-medium text-gray-700 mb-2">Attachments</h3>
                    <div class="flex flex-wrap gap-2">
                        <?php foreach ($attachments as $att): ?>
                        <a href="<?= BASE_URL ?>/uploads/<?= sanitize($att['file_path']) ?>" target="_blank" 
                           class="inline-flex items-center px-3 py-2 bg-gray-100 rounded-lg hover:bg-gray-200 transition text-sm">
                            <i class="fas fa-paperclip mr-2 text-gray-500"></i>
                            <?= sanitize($att['file_name']) ?>
                        </a>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>

                <div class="mt-4 pt-4 border-t text-sm text-gray-500">
                    Submitted on <?= formatDate($complaint['created_at']) ?>
                </div>
            </div>

            <!-- Responses -->
            <div class="bg-white rounded-xl shadow-lg p-6">
                <h2 class="text-lg font-semibold text-gray-800 mb-4">
                    <i class="fas fa-comments mr-2 text-primary"></i>Conversation
                </h2>

                <?php if (empty($responses)): ?>
                <div class="text-center py-8 text-gray-500">
                    <i class="fas fa-comment-slash text-4xl mb-3"></i>
                    <p>No responses yet.</p>
                </div>
                <?php else: ?>
                <div class="space-y-4">
                    <?php foreach ($responses as $resp): ?>
                    <?php if ($resp['is_internal'] && !isStaff()) continue; ?>
                    <?php 
                    // Check if this is a WhatsApp reply (customer message from activity log)
                    $isWhatsAppReply = ($resp['role'] === 'customer');
                    ?>
                    <div class="flex gap-4 <?= $resp['role'] !== 'customer' ? 'flex-row-reverse' : '' ?>">
                        <div class="w-10 h-10 rounded-full flex items-center justify-center flex-shrink-0 <?= $resp['role'] === 'customer' ? 'bg-green-100 text-green-600' : 'bg-purple-100 text-purple-600' ?>">
                            <i class="fab fa-<?= $resp['role'] === 'customer' ? 'whatsapp' : 'user-tie' ?>"></i>
                        </div>
                        <div class="flex-1 <?= $resp['role'] !== 'customer' ? 'text-right' : '' ?>">
                            <div class="inline-block max-w-[85%] <?= $resp['role'] !== 'customer' ? 'text-left' : '' ?>">
                                <div class="<?= $resp['role'] === 'customer' ? 'bg-green-50 border border-green-200' : 'bg-purple-50' ?> rounded-lg p-4 <?= $resp['is_internal'] ? 'border-l-4 border-yellow-400' : '' ?>">
                                    <?php if ($resp['is_internal']): ?>
                                    <span class="text-xs text-yellow-600 font-medium flex items-center gap-1 mb-2">
                                        <i class="fas fa-lock"></i> Internal Note
                                    </span>
                                    <?php elseif ($isWhatsAppReply): ?>
                                    <span class="text-xs text-green-600 font-medium flex items-center gap-1 mb-2">
                                        <i class="fab fa-whatsapp"></i> WhatsApp Reply
                                    </span>
                                    <?php endif; ?>
                                    <p class="text-gray-700"><?= nl2br(sanitize($resp['message'])) ?></p>
                                </div>
                                <p class="text-xs text-gray-500 mt-1">
                                    <?= sanitize($resp['full_name']) ?> • <?= formatDate($resp['created_at']) ?>
                                </p>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>

                <!-- Response Form -->
                <?php if ($canRespond && $complaint['status'] !== 'closed'): ?>
                <form method="POST" class="mt-6 pt-6 border-t">
                    <input type="hidden" name="action" value="respond">
                    <div class="mb-4">
                        <textarea name="message" rows="3" required
                            class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent transition resize-none"
                            placeholder="Type your response..."></textarea>
                    </div>
                    <div class="flex items-center justify-between">
                        <?php if (isStaff()): ?>
                        <label class="flex items-center text-sm text-gray-600">
                            <input type="checkbox" name="is_internal" class="mr-2 rounded text-primary focus:ring-primary">
                            Internal note (not visible to customer)
                        </label>
                        <?php else: ?>
                        <span></span>
                        <?php endif; ?>
                        <button type="submit" class="gradient-bg text-white px-6 py-2 rounded-lg font-medium hover:opacity-90 transition">
                            <i class="fas fa-paper-plane mr-2"></i>Send
                        </button>
                    </div>
                </form>
                <?php endif; ?>
            </div>
        </div>

        <!-- Sidebar -->
        <div class="space-y-6">
            <!-- Branch Info -->
            <div class="bg-white rounded-xl shadow-lg p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">
                    <i class="fas fa-store text-primary mr-2"></i>Branch
                </h3>
                <p class="font-medium text-gray-800"><?= sanitize($complaint['branch_name']) ?></p>
                <?php if ($complaint['branch_address']): ?>
                <p class="text-sm text-gray-500 mt-1"><?= sanitize($complaint['branch_address']) ?></p>
                <?php endif; ?>
            </div>

            <!-- Order Info -->
            <div class="bg-white rounded-xl shadow-lg p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Order Details</h3>
                <div class="space-y-3 text-sm">
                    <div class="flex justify-between">
                        <span class="text-gray-500">Order #</span>
                        <span class="font-medium"><?= sanitize($complaint['order_number']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500">Date</span>
                        <span class="font-medium"><?= date('M d, Y', strtotime($complaint['order_date'])) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500">Amount</span>
                        <span class="font-medium">Rs.<?= number_format($complaint['total_amount'], 2) ?></span>
                    </div>
                </div>
            </div>

            <!-- Complaint Info -->
            <div class="bg-white rounded-xl shadow-lg p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Complaint Info</h3>
                <div class="space-y-3 text-sm">
                    <div class="flex justify-between">
                        <span class="text-gray-500">Department</span>
                        <span class="font-medium"><?= sanitize($complaint['department_name'] ?? 'Pending') ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500">Assigned To</span>
                        <span class="font-medium"><?= sanitize($complaint['assigned_to_name'] ?? 'Unassigned') ?></span>
                    </div>
                    <?php if ($complaint['closed_at']): ?>
                    <div class="flex justify-between">
                        <span class="text-gray-500">Closed By</span>
                        <span class="font-medium"><?= sanitize($complaint['closed_by_name']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500">Closed At</span>
                        <span class="font-medium"><?= formatDate($complaint['closed_at']) ?></span>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Actions -->
            <?php if ($canClose || ($complaint['status'] === 'closed' && ($complaint['customer_id'] == getCurrentUserId() || isAdmin()))): ?>
            <div class="bg-white rounded-xl shadow-lg p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Actions</h3>
                <form method="POST">
                    <?php if ($complaint['status'] !== 'closed' && $canClose): ?>
                    <button type="submit" name="action" value="close" 
                        onclick="return confirm('Are you sure you want to close this complaint?')"
                        class="w-full bg-green-500 text-white py-2 rounded-lg font-medium hover:bg-green-600 transition">
                        <i class="fas fa-check-circle mr-2"></i>Close Complaint
                    </button>
                    <?php elseif ($complaint['status'] === 'closed'): ?>
                    <button type="submit" name="action" value="reopen"
                        onclick="return confirm('Are you sure you want to reopen this complaint?')"
                        class="w-full bg-yellow-500 text-white py-2 rounded-lg font-medium hover:bg-yellow-600 transition">
                        <i class="fas fa-redo mr-2"></i>Reopen Complaint
                    </button>
                    <?php endif; ?>
                </form>
            </div>
            <?php endif; ?>

            <!-- Activity Log -->
            <div class="bg-white rounded-xl shadow-lg p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Activity Log</h3>
                <div class="space-y-3">
                    <?php foreach ($activities as $act): ?>
                    <div class="flex items-start gap-3 text-sm">
                        <div class="w-2 h-2 bg-primary rounded-full mt-2"></div>
                        <div>
                            <p class="text-gray-700">
                                <span class="font-medium"><?= sanitize($act['full_name'] ?? 'System') ?></span>
                                <?= sanitize($act['action']) ?>
                            </p>
                            <p class="text-xs text-gray-500"><?= formatDate($act['created_at']) ?></p>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
