<?php
/**
 * Complaint Submission Form
 * 
 * Accessed via QR code scan from customer invoice
 * URL format: /complaint/submit.php?order=ORDER_NUMBER&branch=BRANCH_ID
 * 
 * Flow:
 * 1. Validate order and branch from URL
 * 2. Ask customer for WhatsApp number and email
 * 3. Send OTP to WhatsApp for verification
 * 4. On verification, create/login customer
 * 5. Show complaint form with detailed issue selection
 * 6. Submit complaint and redirect to dashboard
 */

require_once __DIR__ . '/../config/config.php';

// Order number and Branch ID MUST come from URL (GET parameter via QR code)
$orderNumber = $_GET['order'] ?? '';
$branchId = $_GET['branch'] ?? '';

// Store in session for form submission
if (!empty($orderNumber) && !empty($branchId)) {
    $_SESSION['complaint_order'] = $orderNumber;
    $_SESSION['complaint_branch'] = $branchId;
} else {
    $orderNumber = $_SESSION['complaint_order'] ?? '';
    $branchId = $_SESSION['complaint_branch'] ?? '';
}

if (empty($orderNumber) || empty($branchId)) {
    $pageTitle = 'Invalid Access - Support Center';
    require_once __DIR__ . '/../includes/header.php';
    ?>
    <div class="min-h-[60vh] flex items-center justify-center">
        <div class="bg-white rounded-2xl shadow-xl p-8 max-w-md text-center">
            <div class="w-20 h-20 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-6">
                <i class="fas fa-qrcode text-red-500 text-3xl"></i>
            </div>
            <h1 class="text-2xl font-bold text-gray-800 mb-4">Invalid QR Code</h1>
            <p class="text-gray-600 mb-6">
                Please scan the QR code on your invoice to submit a complaint. The link must contain both order number and branch information.
            </p>
            <p class="text-sm text-gray-500 mb-6">
                If you believe this is an error, please contact our support team.
            </p>
            <a href="<?= BASE_URL ?>/" class="gradient-bg text-white px-6 py-3 rounded-lg font-semibold hover:opacity-90 transition inline-block">
                <i class="fas fa-home mr-2"></i>Go to Home
            </a>
        </div>
    </div>
    <?php
    require_once __DIR__ . '/../includes/footer.php';
    exit;
}

// Fetch order with branch details
$db = getDB();
$stmt = $db->prepare("
    SELECT o.*, 
           b.id as branch_id, b.name as branch_name, b.code as branch_code, b.address as branch_address,
           u.full_name as registered_customer_name, u.email as registered_customer_email
    FROM orders o
    JOIN branches b ON o.branch_id = b.id
    LEFT JOIN users u ON o.customer_id = u.id
    WHERE o.order_number = ? AND o.branch_id = ?
");
$stmt->execute([$orderNumber, $branchId]);
$order = $stmt->fetch();

if (!$order) {
    $pageTitle = 'Order Not Found - Support Center';
    require_once __DIR__ . '/../includes/header.php';
    ?>
    <div class="min-h-[60vh] flex items-center justify-center">
        <div class="bg-white rounded-2xl shadow-xl p-8 max-w-md text-center">
            <div class="w-20 h-20 bg-yellow-100 rounded-full flex items-center justify-center mx-auto mb-6">
                <i class="fas fa-search text-yellow-500 text-3xl"></i>
            </div>
            <h1 class="text-2xl font-bold text-gray-800 mb-4">Order Not Found</h1>
            <p class="text-gray-600 mb-6">
                We couldn't find order <strong><?= sanitize($orderNumber) ?></strong> at this branch. Please verify the QR code on your invoice.
            </p>
            <a href="<?= BASE_URL ?>/" class="gradient-bg text-white px-6 py-3 rounded-lg font-semibold hover:opacity-90 transition inline-block">
                <i class="fas fa-home mr-2"></i>Go to Home
            </a>
        </div>
    </div>
    <?php
    require_once __DIR__ . '/../includes/footer.php';
    exit;
}

// Fetch complaint issues (detailed selection)
$issues = $db->query("SELECT * FROM complaint_issues WHERE is_active = 1 ORDER BY display_order, title")->fetchAll();

$error = '';
$step = 'verify'; // verify, complaint

// Check if customer is already verified
if (isCustomerLoggedIn()) {
    $step = 'complaint';
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'send_otp' || $action === 'check_number') {
        // Check if customer exists or send OTP
        $whatsappNumber = sanitize($_POST['whatsapp_number'] ?? '');
        $email = sanitize($_POST['email'] ?? '');
        
        if (empty($whatsappNumber)) {
            $error = 'Please enter your WhatsApp number.';
        } elseif (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = 'Please enter a valid email address.';
        } else {
            $formattedNumber = WhatsAppOTP::formatPhoneNumber($whatsappNumber);
            
            // Check if customer already exists in database
            $existingCustomer = $db->prepare("SELECT * FROM users WHERE whatsapp_number = ? AND role = 'customer'");
            $existingCustomer->execute([$formattedNumber]);
            $customer = $existingCustomer->fetch();
            
            if ($customer) {
                // Customer exists - log them in directly without OTP
                loginCustomer($customer);
                $step = 'complaint';
            } else {
                // New customer - need OTP verification
                // Check rate limiting
                $rateStmt = $db->prepare("SELECT COUNT(*) as count FROM otp_verifications WHERE whatsapp_number = ? AND created_at > DATE_SUB(NOW(), INTERVAL 10 MINUTE)");
                $rateStmt->execute([$formattedNumber]);
                $rateCheck = $rateStmt->fetch();
                
                if ($rateCheck['count'] >= OTP_MAX_ATTEMPTS) {
                    $error = 'Too many OTP requests. Please try again in 10 minutes.';
                } else {
                    $otpCode = WhatsAppOTP::generateOTP();
                    WhatsAppOTP::storeOTP($db, $formattedNumber, $otpCode);
                    $result = WhatsAppOTP::sendOTP($formattedNumber, $otpCode);
                    
                    if ($result['success']) {
                        $_SESSION['pending_whatsapp'] = $formattedNumber;
                        $_SESSION['pending_email'] = $email;
                        $step = 'verify_otp';
                        
                        // For dev mode, show OTP
                        if (isset($result['dev_otp'])) {
                            $_SESSION['dev_otp'] = $result['dev_otp'];
                        }
                    } else {
                        $error = $result['error'] ?? 'Failed to send OTP. Please try again.';
                    }
                }
            }
        }
    } elseif ($action === 'verify_otp') {
        // Verify OTP
        $otpCode = sanitize($_POST['otp_code'] ?? '');
        $whatsappNumber = $_SESSION['pending_whatsapp'] ?? '';
        $email = $_SESSION['pending_email'] ?? '';
        
        if (empty($otpCode) || empty($whatsappNumber)) {
            $error = 'Please enter the OTP sent to your WhatsApp.';
        } else {
            $result = WhatsAppOTP::verifyOTP($db, $whatsappNumber, $otpCode);
            
            if ($result['success']) {
                // Get or create customer
                $customer = WhatsAppOTP::getOrCreateCustomer($db, $whatsappNumber, $email);
                loginCustomer($customer);
                
                // Clear pending session data
                unset($_SESSION['pending_whatsapp']);
                unset($_SESSION['pending_email']);
                unset($_SESSION['dev_otp']);
                
                $step = 'complaint';
            } else {
                $error = $result['error'] ?? 'Invalid or expired OTP.';
                $step = 'verify_otp';
            }
        }
    } elseif ($action === 'submit_complaint') {
        // Submit complaint
        if (!isCustomerLoggedIn()) {
            $error = 'Please verify your WhatsApp number first.';
        } else {
            // Check if complaint already exists for this order
            $existingStmt = $db->prepare("SELECT id, ticket_number, status FROM complaints WHERE order_id = ?");
            $existingStmt->execute([$order['id']]);
            $existingComplaint = $existingStmt->fetch();
            
            if ($existingComplaint) {
                $error = 'A complaint has already been submitted for this order (Ticket: ' . $existingComplaint['ticket_number'] . '). Each order can only have one complaint.';
            } else {
                $issueId = intval($_POST['issue_id'] ?? 0);
                $additionalDetails = sanitize($_POST['additional_details'] ?? '');
                
                if (empty($issueId)) {
                    $error = 'Please select an issue from the list.';
                } else {
                    // Get issue details for priority and department
                    $issueStmt = $db->prepare("SELECT * FROM complaint_issues WHERE id = ?");
                    $issueStmt->execute([$issueId]);
                    $issue = $issueStmt->fetch();
                    
                    if (!$issue) {
                        $error = 'Invalid issue selected.';
                    } else {
                        $ticketNumber = generateTicketNumber();
                        $accessToken = generateAccessToken();
                        $customerId = getCurrentUserId();
                        $departmentId = $issue['department_id'];
                        $priority = $issue['priority'];
                        
                        // Get branch manager for auto-assignment
                        $branchManager = getBranchManager($db, $order['branch_id']);
                        $assignedTo = $branchManager ? $branchManager['id'] : null;
                        
                        $stmt = $db->prepare("
                            INSERT INTO complaints (ticket_number, access_token, order_id, branch_id, customer_id, issue_id, department_id, additional_details, priority, assigned_to)
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                        ");
                        
                        if ($stmt->execute([$ticketNumber, $accessToken, $order['id'], $order['branch_id'], $customerId, $issueId, $departmentId, $additionalDetails, $priority, $assignedTo])) {
                            $complaintId = $db->lastInsertId();
                            
                            // Log activity
                            $logStmt = $db->prepare("INSERT INTO activity_logs (complaint_id, user_id, action, details) VALUES (?, ?, ?, ?)");
                            $logStmt->execute([$complaintId, $customerId, 'created', 'Complaint submitted for order ' . $orderNumber . ' at branch ' . $order['branch_name']]);
                            
                            if ($assignedTo) {
                                $logStmt->execute([$complaintId, $customerId, 'assigned', 'Auto-assigned to branch manager: ' . $branchManager['full_name']]);
                            }
                            
                            // Handle file upload
                            if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
                                $uploadDir = __DIR__ . '/../uploads/';
                                if (!is_dir($uploadDir)) {
                                    mkdir($uploadDir, 0755, true);
                                }
                                
                                $fileName = $_FILES['attachment']['name'];
                                $fileType = $_FILES['attachment']['type'];
                                $fileSize = $_FILES['attachment']['size'];
                                $fileExt = pathinfo($fileName, PATHINFO_EXTENSION);
                                $newFileName = $ticketNumber . '_' . time() . '.' . $fileExt;
                                $filePath = $uploadDir . $newFileName;
                                
                                if (move_uploaded_file($_FILES['attachment']['tmp_name'], $filePath)) {
                                    $attachStmt = $db->prepare("INSERT INTO complaint_attachments (complaint_id, file_name, file_path, file_type, file_size, uploaded_by) VALUES (?, ?, ?, ?, ?, ?)");
                                    $attachStmt->execute([$complaintId, $fileName, $newFileName, $fileType, $fileSize, $customerId]);
                                }
                            }
                            
                            // Build complaint view URL with access token
                            $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
                            $complaintUrl = $protocol . $_SERVER['HTTP_HOST'] . BASE_URL . '/complaint/view.php?token=' . $accessToken;
                            
                            // Send WhatsApp notification to customer
                            $customerWhatsApp = $_SESSION['whatsapp_number'];
                            $customerMessage = "✅ *Complaint Received*\n\n";
                            $customerMessage .= "Your complaint has been registered successfully.\n\n";
                            $customerMessage .= "📋 *Ticket:* {$ticketNumber}\n";
                            $customerMessage .= "🏪 *Branch:* {$order['branch_name']}\n";
                            $customerMessage .= "📦 *Order:* {$orderNumber}\n";
                            $customerMessage .= "⚠️ *Issue:* {$issue['title']}\n\n";
                            $customerMessage .= "🔗 Track your complaint:\n";
                            $customerMessage .= "{$complaintUrl}\n\n";
                            $customerMessage .= "_Our team will contact you shortly._\n\n- Support Center";
                            
                            $sendResult = WhatsAppOTP::sendMessage($customerWhatsApp, $customerMessage);
                            logWhatsAppMessage($db, $complaintId, $customerWhatsApp, 'complaint_receipt', $customerMessage, null, $sendResult['success']);
                            
                            // Send WhatsApp notification to branch manager
                            if ($branchManager) {
                                $managerMessage = "🔔 *New Complaint Assigned*\n\n";
                                $managerMessage .= "A new complaint has been assigned to you.\n\n";
                                $managerMessage .= "📋 *Ticket:* {$ticketNumber}\n";
                                $managerMessage .= "🏪 *Branch:* {$order['branch_name']}\n";
                                $managerMessage .= "📦 *Order:* {$orderNumber}\n";
                                $managerMessage .= "⚠️ *Issue:* {$issue['title']}\n";
                                $managerMessage .= "🚨 *Priority:* " . ucfirst($priority) . "\n";
                                $managerMessage .= "📱 *Customer:* {$customerWhatsApp}\n\n";
                                $managerMessage .= "_Please review and resolve this complaint._";
                                
                                $sendResult = WhatsAppOTP::sendMessage($branchManager['whatsapp_number'], $managerMessage);
                                logWhatsAppMessage($db, $complaintId, $branchManager['whatsapp_number'], 'manager_notification', $managerMessage, null, $sendResult['success']);
                            }
                            
                            // Clear session data
                            unset($_SESSION['complaint_order']);
                            unset($_SESSION['complaint_branch']);
                            
                            setFlashMessage('success', 'Your complaint has been submitted successfully! Ticket: ' . $ticketNumber);
                            redirect('/customer/dashboard.php');
                        } else {
                            $error = 'Failed to submit complaint. Please try again.';
                        }
                    }
                }
            }
        }
        $step = 'complaint';
    }
}

// Check if we're in OTP verification step
if (isset($_SESSION['pending_whatsapp']) && $step === 'verify') {
    $step = 'verify_otp';
}

$pageTitle = 'Submit Complaint - Support Center';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="max-w-4xl mx-auto">
    <!-- Order & Branch Info Card -->
    <div class="bg-white rounded-xl shadow-lg p-6 mb-6">
        <div class="flex items-center justify-between flex-wrap gap-4 mb-4">
            <div>
                <h2 class="text-lg font-semibold text-gray-800">Order Information</h2>
                <p class="text-gray-500">Submitting complaint for this order</p>
            </div>
            <span class="px-4 py-2 bg-blue-100 text-blue-800 rounded-full font-semibold">
                <?= sanitize($order['order_number']) ?>
            </span>
        </div>
        
        <!-- Branch Info -->
        <div class="bg-gradient-to-r from-primary/10 to-secondary/10 rounded-lg p-4 mb-4">
            <div class="flex items-center gap-3">
                <div class="w-10 h-10 bg-primary rounded-full flex items-center justify-center">
                    <i class="fas fa-store text-white"></i>
                </div>
                <div>
                    <p class="text-sm text-gray-500">Branch</p>
                    <p class="font-semibold text-gray-800"><?= sanitize($order['branch_name']) ?></p>
                    <?php if ($order['branch_address']): ?>
                    <p class="text-xs text-gray-500"><?= sanitize($order['branch_address']) ?></p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div class="bg-gray-50 rounded-lg p-4">
                <p class="text-sm text-gray-500">Order Date</p>
                <p class="font-semibold text-gray-800"><?= date('M d, Y', strtotime($order['order_date'])) ?></p>
            </div>
            <div class="bg-gray-50 rounded-lg p-4">
                <p class="text-sm text-gray-500">Amount</p>
                <p class="font-semibold text-gray-800">Rs.<?= number_format($order['total_amount'], 2) ?></p>
            </div>
            <div class="bg-gray-50 rounded-lg p-4">
                <p class="text-sm text-gray-500">Status</p>
                <p class="font-semibold text-gray-800 capitalize"><?= sanitize($order['status']) ?></p>
            </div>
        </div>
    </div>

    <?php if ($error): ?>
    <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-r-lg mb-6">
        <p><i class="fas fa-exclamation-circle mr-2"></i><?= sanitize($error) ?></p>
    </div>
    <?php endif; ?>

    <?php if ($step === 'verify'): ?>
    <!-- Step 1: WhatsApp Verification -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <div class="text-center mb-6">
            <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <i class="fab fa-whatsapp text-green-500 text-3xl"></i>
            </div>
            <h1 class="text-2xl font-bold text-gray-800">Verify Your WhatsApp</h1>
            <p class="text-gray-500 mt-2">Enter your WhatsApp number to receive a verification code</p>
        </div>

        <form method="POST" class="space-y-6 max-w-md mx-auto">
            <input type="hidden" name="action" value="send_otp">
            
            <div>
                <label for="whatsapp_number" class="block text-sm font-medium text-gray-700 mb-2">
                    <i class="fab fa-whatsapp text-green-500 mr-1"></i>WhatsApp Number *
                </label>
                <input type="tel" id="whatsapp_number" name="whatsapp_number" required
                    class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent transition"
                    placeholder="+1234567890"
                    value="<?= sanitize($_POST['whatsapp_number'] ?? '') ?>">
                <p class="text-xs text-gray-500 mt-1">Enter your WhatsApp number with country code</p>
            </div>

            <div>
                <label for="email" class="block text-sm font-medium text-gray-700 mb-2">
                    <i class="fas fa-envelope text-gray-400 mr-1"></i>Email Address (Optional)
                </label>
                <input type="email" id="email" name="email"
                    class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent transition"
                    placeholder="you@example.com"
                    value="<?= sanitize($_POST['email'] ?? '') ?>">
                <p class="text-xs text-gray-500 mt-1">We'll send complaint updates to this email</p>
            </div>

            <button type="submit" class="w-full bg-green-500 text-white py-3 rounded-lg font-semibold hover:bg-green-600 transition">
                <i class="fab fa-whatsapp mr-2"></i>Send Verification Code
            </button>
        </form>

        <div class="mt-6 pt-6 border-t text-center text-sm text-gray-500">
            <p><i class="fas fa-shield-alt mr-1"></i>Your information is secure and will only be used for complaint tracking</p>
        </div>
    </div>

    <?php elseif ($step === 'verify_otp'): ?>
    <!-- Step 2: OTP Verification -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <div class="text-center mb-6">
            <div class="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <i class="fas fa-key text-blue-500 text-3xl"></i>
            </div>
            <h1 class="text-2xl font-bold text-gray-800">Enter Verification Code</h1>
            <p class="text-gray-500 mt-2">We sent a 6-digit code to your WhatsApp</p>
            <p class="text-sm text-green-600 font-medium mt-1"><?= sanitize($_SESSION['pending_whatsapp'] ?? '') ?></p>
        </div>

        <?php if (isset($_SESSION['dev_otp'])): ?>
        <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6 text-center">
            <p class="text-sm text-yellow-800"><i class="fas fa-code mr-1"></i>DEV MODE - OTP: <strong><?= $_SESSION['dev_otp'] ?></strong></p>
        </div>
        <?php endif; ?>

        <form method="POST" class="space-y-6 max-w-md mx-auto">
            <input type="hidden" name="action" value="verify_otp">
            
            <div>
                <label for="otp_code" class="block text-sm font-medium text-gray-700 mb-2">Verification Code *</label>
                <input type="text" id="otp_code" name="otp_code" required
                    class="w-full px-4 py-4 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent transition text-center text-2xl tracking-widest font-mono"
                    placeholder="000000"
                    maxlength="6"
                    pattern="[0-9]{6}">
                <p class="text-xs text-gray-500 mt-2 text-center">Code expires in <?= OTP_EXPIRY_MINUTES ?> minutes</p>
            </div>

            <button type="submit" class="w-full gradient-bg text-white py-3 rounded-lg font-semibold hover:opacity-90 transition">
                <i class="fas fa-check-circle mr-2"></i>Verify & Continue
            </button>
        </form>

        <div class="mt-6 pt-6 border-t text-center">
            <p class="text-sm text-gray-500 mb-3">Didn't receive the code?</p>
            <form method="POST" class="inline">
                <input type="hidden" name="action" value="send_otp">
                <input type="hidden" name="whatsapp_number" value="<?= sanitize($_SESSION['pending_whatsapp'] ?? '') ?>">
                <input type="hidden" name="email" value="<?= sanitize($_SESSION['pending_email'] ?? '') ?>">
                <button type="submit" class="text-primary hover:underline font-medium">
                    <i class="fas fa-redo mr-1"></i>Resend Code
                </button>
            </form>
        </div>
    </div>

    <?php else: ?>
    <!-- Step 3: Complaint Form -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <div class="flex items-center justify-between mb-6">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">Submit Your Complaint</h1>
                <p class="text-gray-500">Select the issue that best describes your problem</p>
            </div>
            <div class="text-right">
                <p class="text-xs text-gray-500">Verified as</p>
                <p class="text-sm font-medium text-green-600">
                    <i class="fab fa-whatsapp mr-1"></i><?= sanitize($_SESSION['whatsapp_number'] ?? '') ?>
                </p>
            </div>
        </div>

        <form method="POST" enctype="multipart/form-data" class="space-y-6">
            <input type="hidden" name="action" value="submit_complaint">
            
            <!-- Issue Selection -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-3">What issue are you facing? *</label>
                <div class="space-y-3 max-h-96 overflow-y-auto pr-2">
                    <?php foreach ($issues as $issue): ?>
                    <label class="flex items-start p-4 border-2 border-gray-200 rounded-lg cursor-pointer hover:border-primary hover:bg-primary/5 transition issue-option">
                        <input type="radio" name="issue_id" value="<?= $issue['id'] ?>" required
                            class="mt-1 w-4 h-4 text-primary focus:ring-primary border-gray-300"
                            <?= ($_POST['issue_id'] ?? '') == $issue['id'] ? 'checked' : '' ?>>
                        <div class="ml-3 flex-1">
                            <p class="font-medium text-gray-800"><?= sanitize($issue['title']) ?></p>
                            <?php if ($issue['description']): ?>
                            <p class="text-sm text-gray-500 mt-1"><?= sanitize($issue['description']) ?></p>
                            <?php endif; ?>
                            <span class="inline-block mt-2 px-2 py-0.5 rounded text-xs font-medium <?= getPriorityBadgeClass($issue['priority']) ?>">
                                <?= ucfirst($issue['priority']) ?> Priority
                            </span>
                        </div>
                    </label>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Additional Details -->
            <div>
                <label for="additional_details" class="block text-sm font-medium text-gray-700 mb-2">
                    Additional Details (Optional)
                </label>
                <textarea id="additional_details" name="additional_details" rows="4"
                    class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent transition resize-none"
                    placeholder="Provide any additional information that might help us resolve your issue faster..."><?= sanitize($_POST['additional_details'] ?? '') ?></textarea>
            </div>

            <!-- Attachment -->
            <div>
                <label for="attachment" class="block text-sm font-medium text-gray-700 mb-2">Attachment (Optional)</label>
                <div class="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center hover:border-primary transition cursor-pointer" onclick="document.getElementById('attachment').click()">
                    <i class="fas fa-cloud-upload-alt text-gray-400 text-3xl mb-2"></i>
                    <p class="text-gray-600">Click to upload photo or document</p>
                    <p class="text-sm text-gray-400 mt-1">PNG, JPG, PDF up to 5MB</p>
                    <input type="file" id="attachment" name="attachment" class="hidden" accept=".png,.jpg,.jpeg,.pdf,.doc,.docx">
                </div>
                <p id="file-name" class="text-sm text-gray-600 mt-2 hidden"></p>
            </div>

            <div class="flex items-center justify-between pt-4 border-t">
                <a href="<?= BASE_URL ?>/" class="text-gray-600 hover:text-gray-800 transition">
                    <i class="fas fa-arrow-left mr-2"></i>Cancel
                </a>
                <button type="submit" class="gradient-bg text-white px-8 py-3 rounded-lg font-semibold hover:opacity-90 transition">
                    <i class="fas fa-paper-plane mr-2"></i>Submit Complaint
                </button>
            </div>
        </form>
    </div>
    <?php endif; ?>
</div>

<style>
.issue-option:has(input:checked) {
    border-color: #4F46E5;
    background-color: rgba(79, 70, 229, 0.05);
}
</style>

<script>
document.getElementById('attachment')?.addEventListener('change', function(e) {
    const fileName = e.target.files[0]?.name;
    const fileNameEl = document.getElementById('file-name');
    if (fileName) {
        fileNameEl.textContent = 'Selected: ' + fileName;
        fileNameEl.classList.remove('hidden');
    } else {
        fileNameEl.classList.add('hidden');
    }
});

// Auto-focus OTP input
document.getElementById('otp_code')?.focus();
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
