<?php
/**
 * WhatsApp Webhook Handler
 * Receives incoming WhatsApp messages and stores them as complaint responses
 */

require_once __DIR__ . '/../config/config.php';

// Log all incoming requests for debugging
$logFile = __DIR__ . '/../logs/whatsapp-webhook.log';
$logDir = dirname($logFile);
if (!is_dir($logDir)) {
    mkdir($logDir, 0755, true);
}

$requestBody = file_get_contents('php://input');
$timestamp = date('Y-m-d H:i:s');
file_put_contents($logFile, "[$timestamp] Incoming webhook:\n$requestBody\n\n", FILE_APPEND);

// Parse incoming webhook data
$data = json_decode($requestBody, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid JSON']);
    exit;
}

// Handle incoming message from whatsapp-web.js format
if (isset($data['event']) && $data['event'] === 'message' && isset($data['data'])) {
    $messageData = $data['data'];
    
    // Extract message details
    $from = $messageData['from'] ?? '';
    $body = $messageData['body'] ?? '';
    $timestamp = $messageData['timestamp'] ?? time();
    
    // Clean phone number (remove @c.us suffix if present)
    $phoneNumber = str_replace('@c.us', '', $from);
    $phoneNumber = preg_replace('/[^0-9+]/', '', $phoneNumber);
    
    if (empty($phoneNumber) || empty($body)) {
        http_response_code(200);
        echo json_encode(['status' => 'ignored', 'reason' => 'Empty phone or body']);
        exit;
    }
    
    $db = getDB();
    
    // Find customer by WhatsApp number - try multiple formats
    // Format 1: Exact match
    $userStmt = $db->prepare("SELECT id, whatsapp_number FROM users WHERE whatsapp_number = ? AND role = 'customer'");
    $userStmt->execute([$phoneNumber]);
    $user = $userStmt->fetch();
    
    // Format 2: Try with + prefix if not found
    if (!$user && strpos($phoneNumber, '+') !== 0) {
        $userStmt->execute(['+' . $phoneNumber]);
        $user = $userStmt->fetch();
    }
    
    // Format 3: Try without + prefix if not found
    if (!$user && strpos($phoneNumber, '+') === 0) {
        $userStmt->execute([substr($phoneNumber, 1)]);
        $user = $userStmt->fetch();
    }
    
    // Format 4: Try matching last 10 digits (for cases where country code differs)
    if (!$user) {
        $last10 = substr($phoneNumber, -10);
        $userStmt = $db->prepare("SELECT id, whatsapp_number FROM users WHERE whatsapp_number LIKE ? AND role = 'customer' LIMIT 1");
        $userStmt->execute(['%' . $last10]);
        $user = $userStmt->fetch();
    }
    
    if (!$user) {
        file_put_contents($logFile, "[$timestamp] Customer not found for: $phoneNumber (tried multiple formats)\n", FILE_APPEND);
        http_response_code(200);
        echo json_encode(['status' => 'ignored', 'reason' => 'Customer not found', 'phone_tried' => $phoneNumber]);
        exit;
    }
    
    file_put_contents($logFile, "[$timestamp] Customer found: ID={$user['id']}, DB Number={$user['whatsapp_number']}, Incoming={$phoneNumber}\n", FILE_APPEND);
    
    $customerId = $user['id'];
    
    // Find the most recent open complaint for this customer
    $complaintStmt = $db->prepare("
        SELECT c.id, c.ticket_number, c.status 
        FROM complaints c 
        WHERE c.customer_id = ? 
        AND c.status NOT IN ('closed')
        ORDER BY c.created_at DESC 
        LIMIT 1
    ");
    $complaintStmt->execute([$customerId]);
    $complaint = $complaintStmt->fetch();
    
    if (!$complaint) {
        file_put_contents($logFile, "[$timestamp] No open complaint found for customer: $customerId\n", FILE_APPEND);
        http_response_code(200);
        echo json_encode(['status' => 'ignored', 'reason' => 'No open complaint']);
        exit;
    }
    
    // Store the message as a complaint response
    $insertStmt = $db->prepare("
        INSERT INTO complaint_responses (complaint_id, user_id, message, is_internal, created_at) 
        VALUES (?, ?, ?, 0, FROM_UNIXTIME(?))
    ");
    
    if ($insertStmt->execute([$complaint['id'], $customerId, $body, $timestamp])) {
        // Log activity
        $logStmt = $db->prepare("INSERT INTO activity_logs (complaint_id, user_id, action, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$complaint['id'], $customerId, 'customer_replied', 'Customer replied via WhatsApp']);
        
        // Log WhatsApp message
        logWhatsAppMessage($db, $complaint['id'], $phoneNumber, 'other', $body, $customerId, true);
        
        file_put_contents($logFile, "[$timestamp] Message stored for complaint #{$complaint['ticket_number']}\n", FILE_APPEND);
        
        http_response_code(200);
        echo json_encode([
            'status' => 'success',
            'complaint_id' => $complaint['id'],
            'ticket_number' => $complaint['ticket_number']
        ]);
    } else {
        file_put_contents($logFile, "[$timestamp] Failed to store message\n", FILE_APPEND);
        http_response_code(500);
        echo json_encode(['error' => 'Failed to store message']);
    }
    
} else {
    // Unknown webhook format
    http_response_code(200);
    echo json_encode(['status' => 'ignored', 'reason' => 'Unknown format']);
}
