<?php
/**
 * Send OTP API
 * 
 * Sends OTP to customer's WhatsApp number for verification
 */

require_once __DIR__ . '/../config/config.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$whatsappNumber = $input['whatsapp_number'] ?? $_POST['whatsapp_number'] ?? '';

if (empty($whatsappNumber)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'WhatsApp number is required']);
    exit;
}

// Validate phone number format (basic validation)
$formattedNumber = WhatsAppOTP::formatPhoneNumber($whatsappNumber);
if (strlen(preg_replace('/[^0-9]/', '', $formattedNumber)) < 10) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid phone number format']);
    exit;
}

$db = getDB();

// Check rate limiting (max 3 OTPs per number in 10 minutes)
$stmt = $db->prepare("
    SELECT COUNT(*) as count FROM otp_verifications 
    WHERE whatsapp_number = ? AND created_at > DATE_SUB(NOW(), INTERVAL 10 MINUTE)
");
$stmt->execute([$formattedNumber]);
$rateCheck = $stmt->fetch();

if ($rateCheck['count'] >= OTP_MAX_ATTEMPTS) {
    http_response_code(429);
    echo json_encode(['success' => false, 'error' => 'Too many OTP requests. Please try again in 10 minutes.']);
    exit;
}

// Generate and send OTP
$otpCode = WhatsAppOTP::generateOTP();

// Store OTP in database
if (!WhatsAppOTP::storeOTP($db, $formattedNumber, $otpCode)) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Failed to generate OTP']);
    exit;
}

// Send OTP via WhatsApp
$result = WhatsAppOTP::sendOTP($formattedNumber, $otpCode);

if ($result['success']) {
    $response = [
        'success' => true,
        'message' => 'OTP sent to your WhatsApp number',
        'expires_in' => OTP_EXPIRY_MINUTES * 60
    ];
    
    // In dev mode, include OTP for testing
    if (isset($result['dev_otp'])) {
        $response['dev_otp'] = $result['dev_otp'];
        $response['dev_mode'] = true;
    }
    
    echo json_encode($response);
} else {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $result['error']]);
}
