<?php
/**
 * GET Order API
 * 
 * This API is called via QR code scan from customer invoice
 * URL format: /api/get_order.php?order=ORDER_NUMBER&branch=BRANCH_ID
 * 
 * Returns order details including branch information
 */

require_once __DIR__ . '/../config/config.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');

$orderNumber = $_GET['order'] ?? $_GET['order_number'] ?? '';
$branchId = $_GET['branch'] ?? $_GET['branch_id'] ?? '';

if (empty($orderNumber)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Order number is required']);
    exit;
}

if (empty($branchId)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Branch ID is required']);
    exit;
}

$db = getDB();

// Fetch order with branch details
$stmt = $db->prepare("
    SELECT o.*, 
           b.id as branch_id, b.name as branch_name, b.code as branch_code, 
           b.address as branch_address, b.phone as branch_phone,
           u.full_name as registered_customer_name, u.email as registered_customer_email
    FROM orders o
    JOIN branches b ON o.branch_id = b.id
    LEFT JOIN users u ON o.customer_id = u.id
    WHERE o.order_number = ? AND o.branch_id = ?
");
$stmt->execute([$orderNumber, $branchId]);
$order = $stmt->fetch();

if (!$order) {
    http_response_code(404);
    echo json_encode(['success' => false, 'error' => 'Order not found or branch mismatch']);
    exit;
}

// Check if branch is active
$branchStmt = $db->prepare("SELECT is_active FROM branches WHERE id = ?");
$branchStmt->execute([$branchId]);
$branch = $branchStmt->fetch();

if (!$branch || !$branch['is_active']) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Branch is not active']);
    exit;
}

echo json_encode([
    'success' => true,
    'order' => [
        'id' => $order['id'],
        'order_number' => $order['order_number'],
        'customer_name' => $order['customer_name'] ?? $order['registered_customer_name'],
        'order_date' => $order['order_date'],
        'total_amount' => $order['total_amount'],
        'status' => $order['status']
    ],
    'branch' => [
        'id' => $order['branch_id'],
        'name' => $order['branch_name'],
        'code' => $order['branch_code'],
        'address' => $order['branch_address'],
        'phone' => $order['branch_phone']
    ]
]);
