<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn() || !isAdmin()) {
    setFlashMessage('error', 'Access denied. Admin privileges required.');
    redirect('/auth/login.php');
}

$db = getDB();

// Handle user actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $userId = intval($_POST['user_id'] ?? 0);

    if ($action === 'add_user') {
        $fullName = sanitize($_POST['full_name'] ?? '');
        $email = sanitize($_POST['email'] ?? '');
        $whatsappNumber = sanitize($_POST['whatsapp_number'] ?? '');
        $role = sanitize($_POST['role'] ?? '');
        $branchId = intval($_POST['branch_id'] ?? 0) ?: null;
        $password = $_POST['password'] ?? '';
        
        $errors = [];
        
        if (empty($fullName)) $errors[] = 'Full name is required.';
        if (empty($whatsappNumber)) $errors[] = 'WhatsApp number is required.';
        if (!in_array($role, ['customer', 'staff', 'branch_manager', 'admin'])) $errors[] = 'Invalid role selected.';
        if (empty($password) || strlen($password) < 6) $errors[] = 'Password must be at least 6 characters.';
        
        // Branch is mandatory for all roles except admin
        if ($role !== 'admin' && !$branchId) {
            $errors[] = 'Branch assignment is required for ' . ucfirst(str_replace('_', ' ', $role)) . ' role.';
        }
        
        // Check if WhatsApp number already exists
        $checkStmt = $db->prepare("SELECT id FROM users WHERE whatsapp_number = ?");
        $checkStmt->execute([$whatsappNumber]);
        if ($checkStmt->fetch()) {
            $errors[] = 'WhatsApp number already exists.';
        }
        
        // Check if email already exists (if provided)
        if (!empty($email)) {
            $checkStmt = $db->prepare("SELECT id FROM users WHERE email = ?");
            $checkStmt->execute([$email]);
            if ($checkStmt->fetch()) {
                $errors[] = 'Email already exists.';
            }
        }
        
        if (empty($errors)) {
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $db->prepare("INSERT INTO users (full_name, email, whatsapp_number, password, role, branch_id, is_verified, is_active) VALUES (?, ?, ?, ?, ?, ?, 1, 1)");
            
            if ($stmt->execute([$fullName, $email ?: null, $whatsappNumber, $hashedPassword, $role, $branchId])) {
                setFlashMessage('success', 'User added successfully!');
                redirect('/admin/users.php');
            } else {
                $errors[] = 'Failed to add user. Please try again.';
            }
        }
        
        if (!empty($errors)) {
            foreach ($errors as $error) {
                setFlashMessage('error', $error);
            }
        }
    } elseif ($action === 'toggle_status' && $userId) {
        $stmt = $db->prepare("UPDATE users SET is_active = NOT is_active WHERE id = ? AND id != ?");
        $stmt->execute([$userId, getCurrentUserId()]);
        setFlashMessage('success', 'User status updated.');
        redirect('/admin/users.php');
    } elseif ($action === 'change_role' && $userId) {
        $newRole = sanitize($_POST['role'] ?? '');
        $branchId = intval($_POST['branch_id'] ?? 0) ?: null;
        
        if (in_array($newRole, ['customer', 'staff', 'branch_manager', 'admin'])) {
            // Branch is mandatory for all roles except admin
            if ($newRole !== 'admin' && !$branchId) {
                setFlashMessage('error', ucfirst(str_replace('_', ' ', $newRole)) . ' must be assigned to a branch.');
            } else {
                $stmt = $db->prepare("UPDATE users SET role = ?, branch_id = ? WHERE id = ? AND id != ?");
                $stmt->execute([$newRole, $branchId, $userId, getCurrentUserId()]);
                setFlashMessage('success', 'User role updated.');
            }
        }
        redirect('/admin/users.php');
    }
}

// Filters
$roleFilter = $_GET['role'] ?? '';
$search = $_GET['search'] ?? '';

$where = "WHERE 1=1";
$params = [];

if ($roleFilter) {
    $where .= " AND role = ?";
    $params[] = $roleFilter;
}

if ($search) {
    $where .= " AND (full_name LIKE ? OR email LIKE ?)";
    $searchTerm = "%$search%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 15;
$offset = ($page - 1) * $perPage;

$countStmt = $db->prepare("SELECT COUNT(*) as total FROM users $where");
$countStmt->execute($params);
$totalCount = $countStmt->fetch()['total'];
$totalPages = ceil($totalCount / $perPage);

$stmt = $db->prepare("
    SELECT u.*, d.name as department_name, b.name as branch_name,
           (SELECT COUNT(*) FROM complaints WHERE customer_id = u.id) as complaint_count
    FROM users u
    LEFT JOIN departments d ON u.department_id = d.id
    LEFT JOIN branches b ON u.branch_id = b.id
    $where
    ORDER BY u.created_at DESC
    LIMIT $perPage OFFSET $offset
");
$stmt->execute($params);
$users = $stmt->fetchAll();

$departments = $db->query("SELECT * FROM departments ORDER BY name")->fetchAll();
$branches = $db->query("SELECT * FROM branches WHERE is_active = 1 ORDER BY name")->fetchAll();

$pageTitle = 'User Management - Admin';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-6 flex items-center justify-between flex-wrap gap-4">
    <div>
        <h1 class="text-2xl font-bold text-gray-800">User Management</h1>
        <p class="text-gray-500">Manage system users and their roles</p>
    </div>
    <button onclick="openAddUserModal()" class="gradient-bg text-white px-6 py-2 rounded-lg font-medium hover:opacity-90 transition">
        <i class="fas fa-user-plus mr-2"></i>Add New User
    </button>
</div>

<!-- Filters -->
<div class="bg-white rounded-xl shadow-lg p-4 mb-6">
    <form method="GET" class="flex flex-wrap items-center gap-4">
        <div class="flex-1 min-w-[200px]">
            <input type="text" name="search" value="<?= sanitize($search) ?>"
                class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                placeholder="Search by name or email...">
        </div>
        <select name="role" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
            <option value="">All Roles</option>
            <option value="customer" <?= $roleFilter === 'customer' ? 'selected' : '' ?>>Customer</option>
            <option value="staff" <?= $roleFilter === 'staff' ? 'selected' : '' ?>>Staff</option>
            <option value="branch_manager" <?= $roleFilter === 'branch_manager' ? 'selected' : '' ?>>Branch Manager</option>
            <option value="admin" <?= $roleFilter === 'admin' ? 'selected' : '' ?>>Admin</option>
        </select>
        <button type="submit" class="gradient-bg text-white px-6 py-2 rounded-lg font-medium hover:opacity-90 transition">
            <i class="fas fa-filter mr-2"></i>Filter
        </button>
        <?php if ($search || $roleFilter): ?>
        <a href="<?= BASE_URL ?>/admin/users.php" class="text-gray-500 hover:text-gray-700">
            <i class="fas fa-times mr-1"></i>Clear
        </a>
        <?php endif; ?>
    </form>
</div>

<!-- Users Table -->
<div class="bg-white rounded-xl shadow-lg overflow-hidden">
    <?php if (empty($users)): ?>
    <div class="text-center py-12 text-gray-500">
        <i class="fas fa-users text-5xl mb-4"></i>
        <p class="text-lg">No users found</p>
    </div>
    <?php else: ?>
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">User</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">Email</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Role</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">Department</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">Complaints</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($users as $user): ?>
                <tr class="hover:bg-gray-50 transition">
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div class="flex items-center">
                            <div class="w-10 h-10 bg-gray-200 rounded-full flex items-center justify-center mr-3">
                                <i class="fas fa-user text-gray-500"></i>
                            </div>
                            <div>
                                <p class="font-medium text-gray-800"><?= sanitize($user['full_name']) ?></p>
                                <p class="text-xs text-gray-500 md:hidden"><?= sanitize($user['email']) ?></p>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap hidden md:table-cell">
                        <span class="text-gray-600"><?= sanitize($user['email']) ?></span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <?php
                        $roleColors = [
                            'admin' => 'bg-red-100 text-red-800',
                            'branch_manager' => 'bg-purple-100 text-purple-800',
                            'staff' => 'bg-blue-100 text-blue-800',
                            'customer' => 'bg-green-100 text-green-800'
                        ];
                        ?>
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= $roleColors[$user['role']] ?? 'bg-gray-100' ?>">
                            <?= ucfirst(str_replace('_', ' ', $user['role'])) ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap hidden lg:table-cell">
                        <span class="text-gray-600"><?= sanitize($user['branch_name'] ?? ($user['department_name'] ?? 'N/A')) ?></span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap hidden md:table-cell">
                        <span class="text-gray-600"><?= $user['complaint_count'] ?></span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <?php if ($user['is_active']): ?>
                        <span class="px-2 py-1 bg-green-100 text-green-800 rounded-full text-xs font-medium">Active</span>
                        <?php else: ?>
                        <span class="px-2 py-1 bg-red-100 text-red-800 rounded-full text-xs font-medium">Inactive</span>
                        <?php endif; ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-right">
                        <?php if ($user['id'] != getCurrentUserId()): ?>
                        <div class="flex items-center justify-end gap-2">
                            <form method="POST" class="inline">
                                <input type="hidden" name="action" value="toggle_status">
                                <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                <button type="submit" class="text-sm px-2 py-1 rounded <?= $user['is_active'] ? 'bg-red-100 text-red-600 hover:bg-red-200' : 'bg-green-100 text-green-600 hover:bg-green-200' ?> transition">
                                    <?= $user['is_active'] ? 'Deactivate' : 'Activate' ?>
                                </button>
                            </form>
                            <button onclick="openRoleModal(<?= $user['id'] ?>, '<?= $user['role'] ?>', <?= $user['branch_id'] ?? 'null' ?>)" 
                                    class="text-sm px-2 py-1 rounded bg-gray-100 text-gray-600 hover:bg-gray-200 transition">
                                <i class="fas fa-edit mr-1"></i>Edit
                            </button>
                        </div>
                        <?php else: ?>
                        <span class="text-xs text-gray-400">Current User</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
    <div class="px-6 py-4 border-t flex items-center justify-between">
        <p class="text-sm text-gray-500">
            Showing <?= $offset + 1 ?> to <?= min($offset + $perPage, $totalCount) ?> of <?= $totalCount ?> users
        </p>
        <div class="flex gap-2">
            <?php if ($page > 1): ?>
            <a href="?page=<?= $page - 1 ?>&role=<?= urlencode($roleFilter) ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded hover:bg-gray-50 transition">
                <i class="fas fa-chevron-left"></i>
            </a>
            <?php endif; ?>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
            <a href="?page=<?= $i ?>&role=<?= urlencode($roleFilter) ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded <?= $i === $page ? 'bg-primary text-white' : 'hover:bg-gray-50' ?> transition">
                <?= $i ?>
            </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
            <a href="?page=<?= $page + 1 ?>&role=<?= urlencode($roleFilter) ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded hover:bg-gray-50 transition">
                <i class="fas fa-chevron-right"></i>
            </a>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>
    <?php endif; ?>
</div>

<!-- Add User Modal -->
<div id="addUserModal" class="fixed inset-0 bg-black/50 hidden items-center justify-center z-50">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-md mx-4 max-h-[90vh] overflow-y-auto">
        <div class="p-6 border-b sticky top-0 bg-white">
            <h3 class="text-lg font-semibold text-gray-800">Add New User</h3>
        </div>
        <form method="POST" class="p-6 space-y-4">
            <input type="hidden" name="action" value="add_user">
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Full Name *</label>
                <input type="text" name="full_name" required
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                    placeholder="Enter full name">
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">WhatsApp Number *</label>
                <input type="text" name="whatsapp_number" required
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                    placeholder="+92xxxxxxxxxx">
                <p class="text-xs text-gray-500 mt-1">Include country code (e.g., +92)</p>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Email (Optional)</label>
                <input type="email" name="email"
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                    placeholder="user@example.com">
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Password *</label>
                <input type="password" name="password" required minlength="6"
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                    placeholder="Minimum 6 characters">
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Role *</label>
                <select name="role" id="addUserRole" required onchange="toggleAddUserBranchField()"
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                    <option value="">Select Role</option>
                    <option value="customer">Customer</option>
                    <option value="staff">Staff</option>
                    <option value="branch_manager">Branch Manager</option>
                    <option value="admin">Admin</option>
                </select>
            </div>
            
            <div id="addUserBranchField" style="display: none;">
                <label class="block text-sm font-medium text-gray-700 mb-1">Branch *</label>
                <select name="branch_id" id="addUserBranch"
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                    <option value="">Select Branch</option>
                    <?php foreach ($branches as $branch): ?>
                    <option value="<?= $branch['id'] ?>"><?= sanitize($branch['name']) ?></option>
                    <?php endforeach; ?>
                </select>
                <p class="text-xs text-gray-500 mt-1">Required for all roles except Admin</p>
            </div>
            
            <div class="flex gap-3 pt-4">
                <button type="button" onclick="closeAddUserModal()" 
                    class="flex-1 px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition">
                    Cancel
                </button>
                <button type="submit" class="flex-1 gradient-bg text-white px-4 py-2 rounded-lg font-medium hover:opacity-90 transition">
                    <i class="fas fa-user-plus mr-2"></i>Add User
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Role Edit Modal -->
<div id="roleModal" class="fixed inset-0 bg-black/50 hidden items-center justify-center z-50">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-md mx-4">
        <div class="p-6 border-b">
            <h3 class="text-lg font-semibold text-gray-800">Edit User Role</h3>
        </div>
        <form method="POST" class="p-6 space-y-4">
            <input type="hidden" name="action" value="change_role">
            <input type="hidden" name="user_id" id="modalUserId" value="">
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Role *</label>
                <select name="role" id="modalRole" required onchange="toggleBranchField()"
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                    <option value="customer">Customer</option>
                    <option value="staff">Staff</option>
                    <option value="branch_manager">Branch Manager</option>
                    <option value="admin">Admin</option>
                </select>
            </div>
            
            <div id="branchField">
                <label class="block text-sm font-medium text-gray-700 mb-1">Branch *</label>
                <select name="branch_id" id="modalBranch"
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                    <option value="">Select Branch</option>
                    <?php foreach ($branches as $branch): ?>
                    <option value="<?= $branch['id'] ?>"><?= sanitize($branch['name']) ?></option>
                    <?php endforeach; ?>
                </select>
                <p class="text-xs text-gray-500 mt-1">Required for all roles except Admin</p>
            </div>
            
            <div class="flex gap-3 pt-4">
                <button type="button" onclick="closeRoleModal()" 
                    class="flex-1 px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition">
                    Cancel
                </button>
                <button type="submit" class="flex-1 gradient-bg text-white px-4 py-2 rounded-lg font-medium hover:opacity-90 transition">
                    Save Changes
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// Add User Modal Functions
function openAddUserModal() {
    document.getElementById('addUserModal').classList.remove('hidden');
    document.getElementById('addUserModal').classList.add('flex');
}

function closeAddUserModal() {
    document.getElementById('addUserModal').classList.add('hidden');
    document.getElementById('addUserModal').classList.remove('flex');
}

function toggleAddUserBranchField() {
    const role = document.getElementById('addUserRole').value;
    const branchField = document.getElementById('addUserBranchField');
    const branchSelect = document.getElementById('addUserBranch');
    
    if (role && role !== 'admin') {
        branchField.style.display = 'block';
        branchSelect.required = true;
    } else {
        branchField.style.display = 'none';
        branchSelect.required = false;
    }
}

// Edit Role Modal Functions
function openRoleModal(userId, currentRole, currentBranch) {
    document.getElementById('modalUserId').value = userId;
    document.getElementById('modalRole').value = currentRole;
    document.getElementById('modalBranch').value = currentBranch || '';
    toggleBranchField();
    document.getElementById('roleModal').classList.remove('hidden');
    document.getElementById('roleModal').classList.add('flex');
}

function closeRoleModal() {
    document.getElementById('roleModal').classList.add('hidden');
    document.getElementById('roleModal').classList.remove('flex');
}

function toggleBranchField() {
    const role = document.getElementById('modalRole').value;
    const branchField = document.getElementById('branchField');
    const branchSelect = document.getElementById('modalBranch');
    
    if (role && role !== 'admin') {
        branchField.style.display = 'block';
        branchSelect.required = true;
    } else {
        branchField.style.display = 'none';
        branchSelect.required = false;
    }
}

// Close modals on outside click
document.getElementById('addUserModal').addEventListener('click', function(e) {
    if (e.target === this) closeAddUserModal();
});

document.getElementById('roleModal').addEventListener('click', function(e) {
    if (e.target === this) closeRoleModal();
});

// Close modals on Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeAddUserModal();
        closeRoleModal();
    }
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
