<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn() || !isAdmin()) {
    setFlashMessage('error', 'Access denied. Admin privileges required.');
    redirect('/auth/login.php');
}

$db = getDB();

// Filters
$statusFilter = $_GET['status'] ?? '';
$priorityFilter = $_GET['priority'] ?? '';
$departmentFilter = $_GET['department'] ?? '';
$search = $_GET['search'] ?? '';

$where = "WHERE 1=1";
$params = [];

if ($statusFilter) {
    $where .= " AND c.status = ?";
    $params[] = $statusFilter;
}

if ($priorityFilter) {
    $where .= " AND c.priority = ?";
    $params[] = $priorityFilter;
}

if ($departmentFilter) {
    $where .= " AND c.department_id = ?";
    $params[] = $departmentFilter;
}

if ($search) {
    $where .= " AND (c.ticket_number LIKE ? OR i.title LIKE ? OR o.order_number LIKE ? OR u.whatsapp_number LIKE ?)";
    $searchTerm = "%$search%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 15;
$offset = ($page - 1) * $perPage;

// Get total count
$countStmt = $db->prepare("
    SELECT COUNT(*) as total 
    FROM complaints c 
    JOIN orders o ON c.order_id = o.id 
    JOIN users u ON c.customer_id = u.id
    JOIN branches b ON c.branch_id = b.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    $where
");
$countStmt->execute($params);
$totalCount = $countStmt->fetch()['total'];
$totalPages = ceil($totalCount / $perPage);

// Get complaints
$stmt = $db->prepare("
    SELECT c.*, o.order_number, u.whatsapp_number as customer_whatsapp, u.email as customer_email,
           i.title as issue_title, d.name as department_name, b.name as branch_name,
           assigned.full_name as assigned_to_name
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN users u ON c.customer_id = u.id
    JOIN branches b ON c.branch_id = b.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    LEFT JOIN departments d ON c.department_id = d.id
    LEFT JOIN users assigned ON c.assigned_to = assigned.id
    $where
    ORDER BY 
        CASE c.priority 
            WHEN 'urgent' THEN 1 
            WHEN 'high' THEN 2 
            WHEN 'medium' THEN 3 
            ELSE 4 
        END,
        c.created_at DESC
    LIMIT $perPage OFFSET $offset
");
$stmt->execute($params);
$complaints = $stmt->fetchAll();

// Get branches for filter
$branches = $db->query("SELECT * FROM branches WHERE is_active = 1 ORDER BY name")->fetchAll();

// Get departments for filter
$departments = $db->query("SELECT * FROM departments ORDER BY name")->fetchAll();

$pageTitle = 'All Complaints - Admin';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-6 flex items-center justify-between flex-wrap gap-4">
    <div>
        <h1 class="text-2xl font-bold text-gray-800">All Complaints</h1>
        <p class="text-gray-500">Manage and route complaints to departments</p>
    </div>
    <div class="flex gap-2">
        <a href="<?= BASE_URL ?>/admin/export.php" class="bg-green-500 text-white px-4 py-2 rounded-lg hover:bg-green-600 transition">
            <i class="fas fa-download mr-2"></i>Export
        </a>
    </div>
</div>

<!-- Filters -->
<div class="bg-white rounded-xl shadow-lg p-4 mb-6">
    <form method="GET" class="grid grid-cols-1 md:grid-cols-5 gap-4">
        <div>
            <input type="text" name="search" value="<?= sanitize($search) ?>"
                class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                placeholder="Search...">
        </div>
        <select name="status" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
            <option value="">All Status</option>
            <option value="open" <?= $statusFilter === 'open' ? 'selected' : '' ?>>Open</option>
            <option value="in_progress" <?= $statusFilter === 'in_progress' ? 'selected' : '' ?>>In Progress</option>
            <option value="pending_customer" <?= $statusFilter === 'pending_customer' ? 'selected' : '' ?>>Pending Customer</option>
            <option value="resolved" <?= $statusFilter === 'resolved' ? 'selected' : '' ?>>Resolved</option>
            <option value="closed" <?= $statusFilter === 'closed' ? 'selected' : '' ?>>Closed</option>
        </select>
        <select name="priority" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
            <option value="">All Priority</option>
            <option value="urgent" <?= $priorityFilter === 'urgent' ? 'selected' : '' ?>>Urgent</option>
            <option value="high" <?= $priorityFilter === 'high' ? 'selected' : '' ?>>High</option>
            <option value="medium" <?= $priorityFilter === 'medium' ? 'selected' : '' ?>>Medium</option>
            <option value="low" <?= $priorityFilter === 'low' ? 'selected' : '' ?>>Low</option>
        </select>
        <select name="department" class="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
            <option value="">All Departments</option>
            <?php foreach ($departments as $dept): ?>
            <option value="<?= $dept['id'] ?>" <?= $departmentFilter == $dept['id'] ? 'selected' : '' ?>>
                <?= sanitize($dept['name']) ?>
            </option>
            <?php endforeach; ?>
        </select>
        <div class="flex gap-2">
            <button type="submit" class="flex-1 gradient-bg text-white px-4 py-2 rounded-lg font-medium hover:opacity-90 transition">
                <i class="fas fa-filter mr-2"></i>Filter
            </button>
            <?php if ($search || $statusFilter || $priorityFilter || $departmentFilter): ?>
            <a href="<?= BASE_URL ?>/admin/complaints.php" class="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition">
                <i class="fas fa-times"></i>
            </a>
            <?php endif; ?>
        </div>
    </form>
</div>

<!-- Complaints Table -->
<div class="bg-white rounded-xl shadow-lg overflow-hidden">
    <?php if (empty($complaints)): ?>
    <div class="text-center py-12 text-gray-500">
        <i class="fas fa-inbox text-5xl mb-4"></i>
        <p class="text-lg">No complaints found</p>
    </div>
    <?php else: ?>
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ticket</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Issue</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">Customer</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">Branch</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden xl:table-cell">Assigned</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Priority</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">Date</th>
                    <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($complaints as $complaint): ?>
                <tr class="hover:bg-gray-50 transition">
                    <td class="px-4 py-4 whitespace-nowrap">
                        <span class="font-mono text-sm text-primary"><?= sanitize($complaint['ticket_number']) ?></span>
                    </td>
                    <td class="px-4 py-4">
                        <p class="text-gray-800 font-medium line-clamp-1 max-w-[200px]"><?= sanitize($complaint['issue_title'] ?? 'Complaint') ?></p>
                        <p class="text-xs text-gray-500"><?= sanitize($complaint['order_number']) ?></p>
                    </td>
                    <td class="px-4 py-4 whitespace-nowrap hidden md:table-cell">
                        <p class="text-gray-800"><?= sanitize($complaint['customer_whatsapp']) ?></p>
                        <p class="text-xs text-gray-500"><?= sanitize($complaint['customer_email'] ?? '') ?></p>
                    </td>
                    <td class="px-4 py-4 whitespace-nowrap hidden lg:table-cell">
                        <span class="text-gray-600"><?= sanitize($complaint['branch_name']) ?></span>
                    </td>
                    <td class="px-4 py-4 whitespace-nowrap hidden xl:table-cell">
                        <span class="text-gray-600"><?= sanitize($complaint['assigned_to_name'] ?? 'Unassigned') ?></span>
                    </td>
                    <td class="px-4 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getStatusBadgeClass($complaint['status']) ?>">
                            <?= ucfirst(str_replace('_', ' ', $complaint['status'])) ?>
                        </span>
                    </td>
                    <td class="px-4 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getPriorityBadgeClass($complaint['priority']) ?>">
                            <?= ucfirst($complaint['priority']) ?>
                        </span>
                    </td>
                    <td class="px-4 py-4 whitespace-nowrap text-sm text-gray-500 hidden lg:table-cell">
                        <?= date('M d, Y', strtotime($complaint['created_at'])) ?>
                    </td>
                    <td class="px-4 py-4 whitespace-nowrap text-right">
                        <a href="<?= BASE_URL ?>/admin/complaint-detail.php?id=<?= $complaint['id'] ?>" 
                           class="text-primary hover:text-secondary font-medium text-sm">
                            Manage <i class="fas fa-arrow-right ml-1"></i>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
    <div class="px-6 py-4 border-t flex items-center justify-between">
        <p class="text-sm text-gray-500">
            Showing <?= $offset + 1 ?> to <?= min($offset + $perPage, $totalCount) ?> of <?= $totalCount ?> results
        </p>
        <div class="flex gap-2">
            <?php if ($page > 1): ?>
            <a href="?page=<?= $page - 1 ?>&status=<?= urlencode($statusFilter) ?>&priority=<?= urlencode($priorityFilter) ?>&department=<?= urlencode($departmentFilter) ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded hover:bg-gray-50 transition">
                <i class="fas fa-chevron-left"></i>
            </a>
            <?php endif; ?>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
            <a href="?page=<?= $i ?>&status=<?= urlencode($statusFilter) ?>&priority=<?= urlencode($priorityFilter) ?>&department=<?= urlencode($departmentFilter) ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded <?= $i === $page ? 'bg-primary text-white' : 'hover:bg-gray-50' ?> transition">
                <?= $i ?>
            </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
            <a href="?page=<?= $page + 1 ?>&status=<?= urlencode($statusFilter) ?>&priority=<?= urlencode($priorityFilter) ?>&department=<?= urlencode($departmentFilter) ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded hover:bg-gray-50 transition">
                <i class="fas fa-chevron-right"></i>
            </a>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
