<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn() || !isAdmin()) {
    setFlashMessage('error', 'Access denied. Admin privileges required.');
    redirect('/auth/login.php');
}

$complaintId = intval($_GET['id'] ?? 0);

if (!$complaintId) {
    setFlashMessage('error', 'Invalid complaint ID.');
    redirect('/admin/complaints.php');
}

$db = getDB();

// Fetch complaint
$stmt = $db->prepare("
    SELECT c.*, 
           o.order_number, o.order_date, o.total_amount, o.status as order_status,
           u.whatsapp_number as customer_whatsapp, u.email as customer_email,
           i.title as issue_title, i.description as issue_description,
           b.name as branch_name, b.address as branch_address,
           d.name as department_name,
           assigned.full_name as assigned_to_name,
           closer.full_name as closed_by_name
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN users u ON c.customer_id = u.id
    JOIN branches b ON c.branch_id = b.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    LEFT JOIN departments d ON c.department_id = d.id
    LEFT JOIN users assigned ON c.assigned_to = assigned.id
    LEFT JOIN users closer ON c.closed_by = closer.id
    WHERE c.id = ?
");
$stmt->execute([$complaintId]);
$complaint = $stmt->fetch();

if (!$complaint) {
    setFlashMessage('error', 'Complaint not found.');
    redirect('/admin/complaints.php');
}

// Get departments and staff for assignment
$departments = $db->query("SELECT * FROM departments ORDER BY name")->fetchAll();
$staff = $db->query("SELECT * FROM users WHERE role IN ('admin', 'staff') AND is_active = 1 ORDER BY full_name")->fetchAll();
$issues = $db->query("SELECT * FROM complaint_issues WHERE is_active = 1 ORDER BY display_order, title")->fetchAll();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $userId = getCurrentUserId();

    if ($action === 'update') {
        $newStatus = sanitize($_POST['status'] ?? $complaint['status']);
        $newPriority = sanitize($_POST['priority'] ?? $complaint['priority']);
        $newDepartmentId = intval($_POST['department_id'] ?? 0) ?: null;
        $newAssignedTo = intval($_POST['assigned_to'] ?? 0) ?: null;
        $newCategoryId = intval($_POST['category_id'] ?? 0) ?: null;

        $stmt = $db->prepare("
            UPDATE complaints 
            SET status = ?, priority = ?, department_id = ?, assigned_to = ?, updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$newStatus, $newPriority, $newDepartmentId, $newAssignedTo, $complaintId]);

        // Log activity
        $logStmt = $db->prepare("INSERT INTO activity_logs (complaint_id, user_id, action, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$complaintId, $userId, 'updated', 'Complaint details updated']);

        setFlashMessage('success', 'Complaint updated successfully.');
        redirect('/admin/complaint-detail.php?id=' . $complaintId);
    } elseif ($action === 'respond') {
        $message = sanitize($_POST['message'] ?? '');
        $isInternal = isset($_POST['is_internal']);

        if (!empty($message)) {
            $stmt = $db->prepare("INSERT INTO complaint_responses (complaint_id, user_id, message, is_internal) VALUES (?, ?, ?, ?)");
            $stmt->execute([$complaintId, $userId, $message, $isInternal ? 1 : 0]);

            $logStmt = $db->prepare("INSERT INTO activity_logs (complaint_id, user_id, action, details) VALUES (?, ?, ?, ?)");
            $logStmt->execute([$complaintId, $userId, 'responded', $isInternal ? 'Added internal note' : 'Responded to customer']);

            setFlashMessage('success', 'Response added successfully.');
            redirect('/admin/complaint-detail.php?id=' . $complaintId);
        }
    } elseif ($action === 'close') {
        $stmt = $db->prepare("UPDATE complaints SET status = 'closed', closed_by = ?, closed_at = NOW() WHERE id = ?");
        $stmt->execute([$userId, $complaintId]);

        $logStmt = $db->prepare("INSERT INTO activity_logs (complaint_id, user_id, action, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$complaintId, $userId, 'closed', 'Complaint closed by admin']);

        setFlashMessage('success', 'Complaint closed successfully.');
        redirect('/admin/complaint-detail.php?id=' . $complaintId);
    } elseif ($action === 'reopen') {
        $stmt = $db->prepare("UPDATE complaints SET status = 'open', closed_by = NULL, closed_at = NULL WHERE id = ?");
        $stmt->execute([$complaintId]);

        $logStmt = $db->prepare("INSERT INTO activity_logs (complaint_id, user_id, action, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$complaintId, $userId, 'reopened', 'Complaint reopened by admin']);

        setFlashMessage('success', 'Complaint reopened successfully.');
        redirect('/admin/complaint-detail.php?id=' . $complaintId);
    }
}

// Fetch responses
$responses = $db->prepare("
    SELECT r.*, u.full_name, u.role
    FROM complaint_responses r
    JOIN users u ON r.user_id = u.id
    WHERE r.complaint_id = ?
    ORDER BY r.created_at ASC
");
$responses->execute([$complaintId]);
$responses = $responses->fetchAll();

// Fetch attachments
$attachments = $db->prepare("SELECT * FROM complaint_attachments WHERE complaint_id = ?");
$attachments->execute([$complaintId]);
$attachments = $attachments->fetchAll();

// Fetch activity log
$activities = $db->prepare("
    SELECT a.*, u.full_name
    FROM activity_logs a
    LEFT JOIN users u ON a.user_id = u.id
    WHERE a.complaint_id = ?
    ORDER BY a.created_at DESC
");
$activities->execute([$complaintId]);
$activities = $activities->fetchAll();

$pageTitle = 'Manage Complaint - Admin';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-6">
    <nav class="mb-4">
        <ol class="flex items-center space-x-2 text-sm text-gray-500">
            <li><a href="<?= BASE_URL ?>/admin/dashboard.php" class="hover:text-primary">Dashboard</a></li>
            <li><i class="fas fa-chevron-right text-xs"></i></li>
            <li><a href="<?= BASE_URL ?>/admin/complaints.php" class="hover:text-primary">Complaints</a></li>
            <li><i class="fas fa-chevron-right text-xs"></i></li>
            <li class="text-gray-800 font-medium"><?= sanitize($complaint['ticket_number']) ?></li>
        </ol>
    </nav>
</div>

<div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
    <!-- Main Content -->
    <div class="lg:col-span-2 space-y-6">
        <!-- Complaint Header -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <div class="flex items-start justify-between flex-wrap gap-4 mb-4">
                <div>
                    <div class="flex items-center gap-3 mb-2">
                        <span class="px-3 py-1 rounded-full text-sm font-medium <?= getStatusBadgeClass($complaint['status']) ?>">
                            <?= ucfirst(str_replace('_', ' ', $complaint['status'])) ?>
                        </span>
                        <span class="px-3 py-1 rounded-full text-sm font-medium <?= getPriorityBadgeClass($complaint['priority']) ?>">
                            <?= ucfirst($complaint['priority']) ?> Priority
                        </span>
                    </div>
                    <h1 class="text-2xl font-bold text-gray-800"><?= sanitize($complaint['issue_title'] ?? 'Complaint') ?></h1>
                </div>
                <div class="text-right">
                    <p class="text-sm text-gray-500">Ticket #</p>
                    <p class="font-mono font-semibold text-primary"><?= sanitize($complaint['ticket_number']) ?></p>
                </div>
            </div>

            <?php if ($complaint['issue_description']): ?>
            <div class="prose max-w-none text-gray-600 bg-gray-50 rounded-lg p-4 mb-4">
                <p class="text-sm"><i class="fas fa-info-circle mr-1"></i><?= sanitize($complaint['issue_description']) ?></p>
            </div>
            <?php endif; ?>
            
            <?php if ($complaint['additional_details']): ?>
            <div class="prose max-w-none text-gray-700 bg-blue-50 rounded-lg p-4">
                <p class="text-sm font-medium text-gray-600 mb-2">Additional Details from Customer:</p>
                <?= nl2br(sanitize($complaint['additional_details'])) ?>
            </div>
            <?php endif; ?>

            <?php if (!empty($attachments)): ?>
            <div class="mt-4 pt-4 border-t">
                <h3 class="text-sm font-medium text-gray-700 mb-2">Attachments</h3>
                <div class="flex flex-wrap gap-2">
                    <?php foreach ($attachments as $att): ?>
                    <a href="<?= BASE_URL ?>/uploads/<?= sanitize($att['file_path']) ?>" target="_blank" 
                       class="inline-flex items-center px-3 py-2 bg-gray-100 rounded-lg hover:bg-gray-200 transition text-sm">
                        <i class="fas fa-paperclip mr-2 text-gray-500"></i>
                        <?= sanitize($att['file_name']) ?>
                    </a>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>

            <div class="mt-4 pt-4 border-t text-sm text-gray-500">
                Submitted on <?= formatDate($complaint['created_at']) ?>
            </div>
        </div>

        <!-- Responses -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <h2 class="text-lg font-semibold text-gray-800 mb-4">
                <i class="fas fa-comments mr-2 text-primary"></i>Conversation
            </h2>

            <?php if (empty($responses)): ?>
            <div class="text-center py-8 text-gray-500">
                <i class="fas fa-comment-slash text-4xl mb-3"></i>
                <p>No responses yet.</p>
            </div>
            <?php else: ?>
            <div class="space-y-4 max-h-96 overflow-y-auto">
                <?php foreach ($responses as $resp): ?>
                <div class="flex gap-4 <?= $resp['role'] !== 'customer' ? 'flex-row-reverse' : '' ?>">
                    <div class="w-10 h-10 rounded-full flex items-center justify-center flex-shrink-0 <?= $resp['role'] === 'customer' ? 'bg-blue-100 text-blue-600' : 'bg-purple-100 text-purple-600' ?>">
                        <i class="fas fa-<?= $resp['role'] === 'customer' ? 'user' : 'headset' ?>"></i>
                    </div>
                    <div class="flex-1 <?= $resp['role'] !== 'customer' ? 'text-right' : '' ?>">
                        <div class="inline-block max-w-[85%] <?= $resp['role'] !== 'customer' ? 'text-left' : '' ?>">
                            <div class="<?= $resp['role'] === 'customer' ? 'bg-blue-50' : 'bg-purple-50' ?> rounded-lg p-4 <?= $resp['is_internal'] ? 'border-l-4 border-yellow-400' : '' ?>">
                                <?php if ($resp['is_internal']): ?>
                                <span class="text-xs text-yellow-600 font-medium">Internal Note</span>
                                <?php endif; ?>
                                <p class="text-gray-700"><?= nl2br(sanitize($resp['message'])) ?></p>
                            </div>
                            <p class="text-xs text-gray-500 mt-1">
                                <?= sanitize($resp['full_name']) ?> (<?= ucfirst($resp['role']) ?>) • <?= formatDate($resp['created_at']) ?>
                            </p>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <!-- Response Form -->
            <?php if ($complaint['status'] !== 'closed'): ?>
            <form method="POST" class="mt-6 pt-6 border-t">
                <input type="hidden" name="action" value="respond">
                <div class="mb-4">
                    <textarea name="message" rows="3" required
                        class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent transition resize-none"
                        placeholder="Type your response..."></textarea>
                </div>
                <div class="flex items-center justify-between">
                    <label class="flex items-center text-sm text-gray-600">
                        <input type="checkbox" name="is_internal" class="mr-2 rounded text-primary focus:ring-primary">
                        Internal note (not visible to customer)
                    </label>
                    <button type="submit" class="gradient-bg text-white px-6 py-2 rounded-lg font-medium hover:opacity-90 transition">
                        <i class="fas fa-paper-plane mr-2"></i>Send
                    </button>
                </div>
            </form>
            <?php endif; ?>
        </div>
    </div>

    <!-- Sidebar -->
    <div class="space-y-6">
        <!-- Branch Info -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">
                <i class="fas fa-store text-primary mr-2"></i>Branch
            </h3>
            <p class="font-medium text-gray-800"><?= sanitize($complaint['branch_name']) ?></p>
            <?php if ($complaint['branch_address']): ?>
            <p class="text-sm text-gray-500 mt-1"><?= sanitize($complaint['branch_address']) ?></p>
            <?php endif; ?>
        </div>

        <!-- Customer Info -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Customer</h3>
            <div class="flex items-center gap-4 mb-4">
                <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                    <i class="fab fa-whatsapp text-green-500 text-xl"></i>
                </div>
                <div>
                    <p class="font-medium text-gray-800"><?= sanitize($complaint['customer_whatsapp']) ?></p>
                    <?php if ($complaint['customer_email']): ?>
                    <p class="text-sm text-gray-500"><?= sanitize($complaint['customer_email']) ?></p>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Order Info -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Order Details</h3>
            <div class="space-y-3 text-sm">
                <div class="flex justify-between">
                    <span class="text-gray-500">Order #</span>
                    <span class="font-medium"><?= sanitize($complaint['order_number']) ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-500">Date</span>
                    <span class="font-medium"><?= date('M d, Y', strtotime($complaint['order_date'])) ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-500">Amount</span>
                    <span class="font-medium">$<?= number_format($complaint['total_amount'], 2) ?></span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-500">Status</span>
                    <span class="font-medium capitalize"><?= sanitize($complaint['order_status']) ?></span>
                </div>
            </div>
        </div>

        <!-- Update Complaint -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Update Complaint</h3>
            <form method="POST" class="space-y-4">
                <input type="hidden" name="action" value="update">
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                    <select name="status" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                        <option value="open" <?= $complaint['status'] === 'open' ? 'selected' : '' ?>>Open</option>
                        <option value="in_progress" <?= $complaint['status'] === 'in_progress' ? 'selected' : '' ?>>In Progress</option>
                        <option value="pending_customer" <?= $complaint['status'] === 'pending_customer' ? 'selected' : '' ?>>Pending Customer</option>
                        <option value="resolved" <?= $complaint['status'] === 'resolved' ? 'selected' : '' ?>>Resolved</option>
                    </select>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Priority</label>
                    <select name="priority" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                        <option value="low" <?= $complaint['priority'] === 'low' ? 'selected' : '' ?>>Low</option>
                        <option value="medium" <?= $complaint['priority'] === 'medium' ? 'selected' : '' ?>>Medium</option>
                        <option value="high" <?= $complaint['priority'] === 'high' ? 'selected' : '' ?>>High</option>
                        <option value="urgent" <?= $complaint['priority'] === 'urgent' ? 'selected' : '' ?>>Urgent</option>
                    </select>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Department</label>
                    <select name="department_id" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                        <option value="">Select Department</option>
                        <?php foreach ($departments as $dept): ?>
                        <option value="<?= $dept['id'] ?>" <?= $complaint['department_id'] == $dept['id'] ? 'selected' : '' ?>>
                            <?= sanitize($dept['name']) ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Assign To</label>
                    <select name="assigned_to" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                        <option value="">Unassigned</option>
                        <?php foreach ($staff as $s): ?>
                        <option value="<?= $s['id'] ?>" <?= $complaint['assigned_to'] == $s['id'] ? 'selected' : '' ?>>
                            <?= sanitize($s['full_name']) ?> (<?= ucfirst($s['role']) ?>)
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <button type="submit" class="w-full gradient-bg text-white py-2 rounded-lg font-medium hover:opacity-90 transition">
                    <i class="fas fa-save mr-2"></i>Update
                </button>
            </form>
        </div>

        <!-- Actions -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Actions</h3>
            <form method="POST" class="space-y-3">
                <?php if ($complaint['status'] !== 'closed'): ?>
                <button type="submit" name="action" value="close" 
                    onclick="return confirm('Are you sure you want to close this complaint?')"
                    class="w-full bg-green-500 text-white py-2 rounded-lg font-medium hover:bg-green-600 transition">
                    <i class="fas fa-check-circle mr-2"></i>Close Complaint
                </button>
                <?php else: ?>
                <button type="submit" name="action" value="reopen"
                    onclick="return confirm('Are you sure you want to reopen this complaint?')"
                    class="w-full bg-yellow-500 text-white py-2 rounded-lg font-medium hover:bg-yellow-600 transition">
                    <i class="fas fa-redo mr-2"></i>Reopen Complaint
                </button>
                <?php endif; ?>
                
                <a href="<?= BASE_URL ?>/complaint/view.php?ticket=<?= sanitize($complaint['ticket_number']) ?>" 
                   target="_blank"
                   class="block w-full text-center bg-gray-100 text-gray-700 py-2 rounded-lg font-medium hover:bg-gray-200 transition">
                    <i class="fas fa-external-link-alt mr-2"></i>View as Customer
                </a>
            </form>
        </div>

        <!-- Activity Log -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Activity Log</h3>
            <div class="space-y-3 max-h-64 overflow-y-auto">
                <?php foreach ($activities as $act): ?>
                <div class="flex items-start gap-3 text-sm">
                    <div class="w-2 h-2 bg-primary rounded-full mt-2"></div>
                    <div>
                        <p class="text-gray-700">
                            <span class="font-medium"><?= sanitize($act['full_name'] ?? 'System') ?></span>
                            <?= sanitize($act['action']) ?>
                        </p>
                        <p class="text-xs text-gray-500"><?= formatDate($act['created_at']) ?></p>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
