<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn() || !isAdmin()) {
    setFlashMessage('error', 'Access denied. Admin privileges required.');
    redirect('/auth/login.php');
}

$db = getDB();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add') {
        $name = sanitize($_POST['name'] ?? '');
        $code = sanitize($_POST['code'] ?? '');
        $address = sanitize($_POST['address'] ?? '');
        $phone = sanitize($_POST['phone'] ?? '');
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        
        if (empty($name) || empty($code)) {
            setFlashMessage('error', 'Branch name and code are required.');
        } else {
            // Check if code already exists
            $checkStmt = $db->prepare("SELECT id FROM branches WHERE code = ?");
            $checkStmt->execute([$code]);
            if ($checkStmt->fetch()) {
                setFlashMessage('error', 'Branch code already exists.');
            } else {
                $stmt = $db->prepare("INSERT INTO branches (name, code, address, phone, is_active) VALUES (?, ?, ?, ?, ?)");
                $stmt->execute([$name, $code, $address, $phone, $isActive]);
                setFlashMessage('success', 'Branch added successfully.');
            }
        }
        redirect('/admin/branches.php');
    } elseif ($action === 'edit') {
        $id = intval($_POST['id'] ?? 0);
        $name = sanitize($_POST['name'] ?? '');
        $code = sanitize($_POST['code'] ?? '');
        $address = sanitize($_POST['address'] ?? '');
        $phone = sanitize($_POST['phone'] ?? '');
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        
        if (empty($name) || empty($code) || !$id) {
            setFlashMessage('error', 'Branch name and code are required.');
        } else {
            // Check if code already exists for another branch
            $checkStmt = $db->prepare("SELECT id FROM branches WHERE code = ? AND id != ?");
            $checkStmt->execute([$code, $id]);
            if ($checkStmt->fetch()) {
                setFlashMessage('error', 'Branch code already exists.');
            } else {
                $stmt = $db->prepare("UPDATE branches SET name = ?, code = ?, address = ?, phone = ?, is_active = ? WHERE id = ?");
                $stmt->execute([$name, $code, $address, $phone, $isActive, $id]);
                setFlashMessage('success', 'Branch updated successfully.');
            }
        }
        redirect('/admin/branches.php');
    } elseif ($action === 'delete') {
        $id = intval($_POST['id'] ?? 0);
        
        if ($id) {
            // Check if branch has complaints
            $checkStmt = $db->prepare("SELECT COUNT(*) as count FROM complaints WHERE branch_id = ?");
            $checkStmt->execute([$id]);
            $count = $checkStmt->fetch()['count'];
            
            if ($count > 0) {
                setFlashMessage('error', 'Cannot delete branch with existing complaints. Deactivate it instead.');
            } else {
                $stmt = $db->prepare("DELETE FROM branches WHERE id = ?");
                $stmt->execute([$id]);
                setFlashMessage('success', 'Branch deleted successfully.');
            }
        }
        redirect('/admin/branches.php');
    } elseif ($action === 'toggle') {
        $id = intval($_POST['id'] ?? 0);
        if ($id) {
            $stmt = $db->prepare("UPDATE branches SET is_active = NOT is_active WHERE id = ?");
            $stmt->execute([$id]);
            setFlashMessage('success', 'Branch status updated.');
        }
        redirect('/admin/branches.php');
    }
}

// Fetch all branches
$branches = $db->query("
    SELECT b.*, 
           (SELECT COUNT(*) FROM complaints WHERE branch_id = b.id) as complaint_count,
           (SELECT COUNT(*) FROM orders WHERE branch_id = b.id) as order_count
    FROM branches b 
    ORDER BY b.name
")->fetchAll();

$pageTitle = 'Manage Branches - Admin';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-6 flex items-center justify-between">
    <div>
        <h1 class="text-2xl font-bold text-gray-800">Manage Branches</h1>
        <p class="text-gray-500">Add, edit, and manage branch locations</p>
    </div>
    <button onclick="openAddModal()" class="gradient-bg text-white px-6 py-2 rounded-lg font-medium hover:opacity-90 transition">
        <i class="fas fa-plus mr-2"></i>Add Branch
    </button>
</div>

<!-- Branches Table -->
<div class="bg-white rounded-xl shadow-lg overflow-hidden">
    <?php if (empty($branches)): ?>
    <div class="text-center py-12 text-gray-500">
        <i class="fas fa-store text-5xl mb-4"></i>
        <p class="text-lg">No branches found</p>
        <p class="text-sm mt-2">Click "Add Branch" to create your first branch.</p>
    </div>
    <?php else: ?>
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Branch</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Code</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">Address</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">Phone</th>
                    <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase">Complaints</th>
                    <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($branches as $branch): ?>
                <tr class="hover:bg-gray-50 transition">
                    <td class="px-6 py-4">
                        <div class="flex items-center">
                            <div class="w-10 h-10 bg-primary/10 rounded-full flex items-center justify-center mr-3">
                                <i class="fas fa-store text-primary"></i>
                            </div>
                            <span class="font-medium text-gray-800"><?= sanitize($branch['name']) ?></span>
                        </div>
                    </td>
                    <td class="px-6 py-4">
                        <span class="font-mono text-sm bg-gray-100 px-2 py-1 rounded"><?= sanitize($branch['code']) ?></span>
                    </td>
                    <td class="px-6 py-4 hidden md:table-cell">
                        <span class="text-gray-600 text-sm"><?= sanitize($branch['address'] ?: '-') ?></span>
                    </td>
                    <td class="px-6 py-4 hidden lg:table-cell">
                        <span class="text-gray-600 text-sm"><?= sanitize($branch['phone'] ?: '-') ?></span>
                    </td>
                    <td class="px-6 py-4 text-center">
                        <span class="px-2 py-1 bg-blue-100 text-blue-800 rounded-full text-sm font-medium">
                            <?= $branch['complaint_count'] ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 text-center">
                        <form method="POST" class="inline">
                            <input type="hidden" name="action" value="toggle">
                            <input type="hidden" name="id" value="<?= $branch['id'] ?>">
                            <button type="submit" class="<?= $branch['is_active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' ?> px-3 py-1 rounded-full text-xs font-medium hover:opacity-80 transition">
                                <?= $branch['is_active'] ? 'Active' : 'Inactive' ?>
                            </button>
                        </form>
                    </td>
                    <td class="px-6 py-4 text-right">
                        <button onclick="openEditModal(<?= htmlspecialchars(json_encode($branch)) ?>)" 
                                class="text-blue-600 hover:text-blue-800 mr-3">
                            <i class="fas fa-edit"></i>
                        </button>
                        <?php if ($branch['complaint_count'] == 0): ?>
                        <form method="POST" class="inline" onsubmit="return confirm('Are you sure you want to delete this branch?')">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="id" value="<?= $branch['id'] ?>">
                            <button type="submit" class="text-red-600 hover:text-red-800">
                                <i class="fas fa-trash"></i>
                            </button>
                        </form>
                        <?php else: ?>
                        <span class="text-gray-400 cursor-not-allowed" title="Cannot delete branch with complaints">
                            <i class="fas fa-trash"></i>
                        </span>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php endif; ?>
</div>

<!-- Add/Edit Modal -->
<div id="branchModal" class="fixed inset-0 bg-black/50 hidden items-center justify-center z-50">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-md mx-4">
        <div class="p-6 border-b">
            <h3 id="modalTitle" class="text-lg font-semibold text-gray-800">Add Branch</h3>
        </div>
        <form id="branchForm" method="POST" class="p-6 space-y-4">
            <input type="hidden" name="action" id="formAction" value="add">
            <input type="hidden" name="id" id="branchId" value="">
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Branch Name *</label>
                <input type="text" name="name" id="branchName" required
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                    placeholder="e.g., Main Branch">
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Branch Code *</label>
                <input type="text" name="code" id="branchCode" required
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                    placeholder="e.g., MAIN">
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Address</label>
                <textarea name="address" id="branchAddress" rows="2"
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                    placeholder="Full address"></textarea>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Phone</label>
                <input type="text" name="phone" id="branchPhone"
                    class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                    placeholder="e.g., +1234567890">
            </div>
            
            <div>
                <label class="flex items-center">
                    <input type="checkbox" name="is_active" id="branchActive" checked
                        class="w-4 h-4 text-primary border-gray-300 rounded focus:ring-primary">
                    <span class="ml-2 text-sm text-gray-700">Active</span>
                </label>
            </div>
            
            <div class="flex gap-3 pt-4">
                <button type="button" onclick="closeModal()" 
                    class="flex-1 px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition">
                    Cancel
                </button>
                <button type="submit" class="flex-1 gradient-bg text-white px-4 py-2 rounded-lg font-medium hover:opacity-90 transition">
                    Save Branch
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openAddModal() {
    document.getElementById('modalTitle').textContent = 'Add Branch';
    document.getElementById('formAction').value = 'add';
    document.getElementById('branchId').value = '';
    document.getElementById('branchName').value = '';
    document.getElementById('branchCode').value = '';
    document.getElementById('branchAddress').value = '';
    document.getElementById('branchPhone').value = '';
    document.getElementById('branchActive').checked = true;
    document.getElementById('branchModal').classList.remove('hidden');
    document.getElementById('branchModal').classList.add('flex');
}

function openEditModal(branch) {
    document.getElementById('modalTitle').textContent = 'Edit Branch';
    document.getElementById('formAction').value = 'edit';
    document.getElementById('branchId').value = branch.id;
    document.getElementById('branchName').value = branch.name;
    document.getElementById('branchCode').value = branch.code;
    document.getElementById('branchAddress').value = branch.address || '';
    document.getElementById('branchPhone').value = branch.phone || '';
    document.getElementById('branchActive').checked = branch.is_active == 1;
    document.getElementById('branchModal').classList.remove('hidden');
    document.getElementById('branchModal').classList.add('flex');
}

function closeModal() {
    document.getElementById('branchModal').classList.add('hidden');
    document.getElementById('branchModal').classList.remove('flex');
}

// Close modal on outside click
document.getElementById('branchModal').addEventListener('click', function(e) {
    if (e.target === this) closeModal();
});

// Close modal on Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') closeModal();
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
