# Two-Way WhatsApp Integration Guide

## Overview
This system now supports **two-way WhatsApp conversations** between customers and managers:
- **Managers** send messages from the web panel
- **Customers** receive messages in WhatsApp and can reply directly
- **Customer replies** automatically appear in the manager's web panel

## How It Works

### 1. Manager Sends Message (Web → WhatsApp)
1. Manager views complaint in web panel
2. Manager types response and clicks "Send Response"
3. Message is saved to database
4. WhatsApp notification sent to customer with the message
5. Customer receives message in their WhatsApp

### 2. Customer Replies (WhatsApp → Web)
1. Customer replies to the WhatsApp message
2. WhatsApp service receives the incoming message
3. Message is forwarded to webhook endpoint
4. Webhook identifies the customer and their open complaint
5. Reply is stored as a complaint response
6. Manager sees the reply in the web panel (with green WhatsApp badge)

## Setup Instructions

### Step 1: Install Dependencies
```bash
cd whatsapp-service
npm install
```

This will install the new `axios` dependency needed for webhooks.

### Step 2: Configure Webhook URL
Edit `whatsapp-service/.env` (create if doesn't exist):
```
WEBHOOK_URL=http://your-domain.com/api/whatsapp-webhook.php
```

For local development:
```
WEBHOOK_URL=http://localhost/api/whatsapp-webhook.php
```

### Step 3: Start WhatsApp Service
```bash
cd whatsapp-service
npm start
```

Scan the QR code with WhatsApp to connect.

### Step 4: Verify Webhook
The webhook endpoint is at: `/api/whatsapp-webhook.php`

Check logs at: `/logs/whatsapp-webhook.log`

## Features

### Visual Indicators
- **Customer WhatsApp replies**: Green background with WhatsApp icon
- **Manager responses**: Purple background with staff icon
- **Internal notes**: Yellow border (only visible to staff)

### Automatic Complaint Matching
- System finds the customer's most recent open complaint
- Replies are automatically linked to the correct complaint
- No manual linking required

### Message Logging
All WhatsApp messages (sent and received) are logged in the `whatsapp_messages` table for audit purposes.

## Troubleshooting

### Branch Manager Not Receiving Notifications

**Check 1: Verify Branch Manager Exists**
```sql
SELECT * FROM users WHERE role = 'branch_manager' AND branch_id = YOUR_BRANCH_ID;
```

**Check 2: Verify WhatsApp Number Format**
Branch manager's WhatsApp number must include country code (e.g., +92xxxxxxxxxx)

**Check 3: Check WhatsApp Service Status**
```bash
curl http://localhost:3001/health
```

Should return: `{"status":"ok","connected":true}`

**Check 4: Check Message Logs**
```sql
SELECT * FROM whatsapp_messages WHERE message_type = 'manager_notification' ORDER BY created_at DESC LIMIT 10;
```

**Check 5: Verify Branch Assignment**
```sql
SELECT c.*, u.full_name as manager_name, u.whatsapp_number as manager_phone
FROM complaints c
LEFT JOIN users u ON u.branch_id = c.branch_id AND u.role = 'branch_manager'
WHERE c.id = YOUR_COMPLAINT_ID;
```

### Customer Replies Not Appearing

**Check 1: Verify Webhook Logs**
```bash
tail -f logs/whatsapp-webhook.log
```

**Check 2: Test Webhook Manually**
```bash
curl -X POST http://localhost/api/whatsapp-webhook.php \
  -H "Content-Type: application/json" \
  -d '{
    "event": "message",
    "data": {
      "from": "92xxxxxxxxxx@c.us",
      "body": "Test message",
      "timestamp": 1234567890
    }
  }'
```

**Check 3: Verify Customer Has Open Complaint**
```sql
SELECT * FROM complaints WHERE customer_id = YOUR_CUSTOMER_ID AND status != 'closed';
```

## Production Deployment

### 1. Deploy WhatsApp Service
Follow instructions in `whatsapp-service/DEPLOYMENT.md`

### 2. Update Webhook URL
Set the production webhook URL in environment variables:
```
WEBHOOK_URL=https://your-domain.com/api/whatsapp-webhook.php
```

### 3. Secure Webhook Endpoint
Add authentication to `/api/whatsapp-webhook.php`:
```php
// Add at the top of webhook file
$secret = 'your-secret-key';
$receivedSecret = $_SERVER['HTTP_X_WEBHOOK_SECRET'] ?? '';
if ($receivedSecret !== $secret) {
    http_response_code(401);
    exit;
}
```

Update server.js to send secret:
```javascript
await axios.post(webhookUrl, webhookData, {
    headers: { 
        'Content-Type': 'application/json',
        'X-Webhook-Secret': process.env.WEBHOOK_SECRET || 'your-secret-key'
    }
});
```

## Database Schema

### whatsapp_messages Table
Logs all WhatsApp communications:
- `complaint_id`: Linked complaint
- `recipient_number`: Phone number
- `message_type`: otp, complaint_receipt, manager_notification, customer_update, other
- `message_content`: Full message text
- `is_sent`: Success status
- `sent_at`: Timestamp

### complaint_responses Table
Stores all conversation messages:
- `complaint_id`: Linked complaint
- `user_id`: Who sent the message
- `message`: Message content
- `is_internal`: Internal note flag
- `created_at`: Timestamp

## API Endpoints

### WhatsApp Service
- `GET /health` - Health check
- `GET /status` - Connection status
- `POST /send-otp` - Send OTP
- `POST /send-message` - Send custom message
- `POST /logout` - Logout and reset

### Webhook
- `POST /api/whatsapp-webhook.php` - Receive incoming WhatsApp messages

## Support

For issues or questions, check:
1. WhatsApp service logs: `whatsapp-service/` console output
2. Webhook logs: `logs/whatsapp-webhook.log`
3. Database logs: `whatsapp_messages` table
4. PHP error logs: Check your server's PHP error log
